package com.atlassian.plugin.cache.filecache.impl;

import com.atlassian.plugin.cache.filecache.FileCache;
import com.atlassian.plugin.cache.filecache.FileCacheStreamProvider;
import com.atlassian.plugin.servlet.DownloadException;
import com.google.common.cache.Cache;
import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheLoader;
import com.google.common.cache.RemovalListener;
import com.google.common.cache.RemovalNotification;

import java.io.File;
import java.io.IOException;
import java.io.OutputStream;
import java.util.ArrayList;
import java.util.Collection;
import java.util.concurrent.atomic.AtomicLong;

/**
 * Implements a thread-safe FileCache.
 * <p/>
 * This implementation uses a computing map LRU map to create missing entries, and an eviction listener
 * to identify files that need to be deleted.
 * <p/>
 * Concurrent control to the contents of each cache entry is implemented in {@link CachedFile}
 *
 * @since v2.13
 */
public class FileCacheImpl<K> implements FileCache<K> {

    static final String EXT = ".cachedfile";

    private final Cache<K, CachedFile> cache;
    private final File tmpDir;
    private final AtomicLong filenameCounter;

    /**
     * @param tmpDir            directory to store cache contents. Files in this dir will be deleted by constructor
     * @param maxSize           size of the LRU cache. zero means evict-immediately
     * @param filenameCounter   counter for cache filenames. Passed in to allow multiple instances to share the same counter.
     */
    public FileCacheImpl(File tmpDir, int maxSize, AtomicLong filenameCounter) throws IOException {
        if (maxSize < 0) {
            throw new IllegalArgumentException("Max files can not be less than zero");
        }
        initDirectory(tmpDir);

        this.tmpDir = tmpDir;

        cache = CacheBuilder.newBuilder()
                .maximumSize(maxSize)
                .removalListener(new RemovalListener<K, CachedFile>() {
                    @Override
                    public void onRemoval(RemovalNotification<K, CachedFile> notification)
                    {
                        onEviction(notification.getValue());
                    }
                })
                .build(new CacheLoader<K, CachedFile>()
                {
                    @Override
                    public CachedFile load(K key) throws Exception
                    {
                        return newCachedFile();
                    }
                });

        this.filenameCounter = filenameCounter;
    }

    private static void initDirectory(File tmpDir) throws IOException {
        tmpDir.mkdirs();
        File[] files = tmpDir.listFiles();
        if (files != null) {
            for (File f : files) {
                if (f.getName().toLowerCase().endsWith(EXT) && !f.delete()) {
                    throw new IOException("Could not delete existing cache file " + f);
                }
            }
        }

        if (!tmpDir.isDirectory()) {
            throw new IOException("Could not create tmp directory " + tmpDir);
        }
    }

    @Override
    public void stream(K key, OutputStream dest, FileCacheStreamProvider input) throws DownloadException {
        final CachedFile cachedFile = cache.getUnchecked(key);
        // Call stream() ASAP, even though `cachedFile` may already be evicted by the time we call it.
        // That's okay, though, it still streams (just not from the cache).
        cachedFile.stream(dest, input);
    }

    @Override
    public void clear() {
        // between the call to .values() and .clear(), further cache entries may be added.
        // In that case, we will never call .delete() on those files, and they will be orhpaned.
        // That's not so bad.
        
        Collection<CachedFile> cachedFiles = new ArrayList<CachedFile>(cache.asMap().values());
        cache.invalidateAll();
        
        for (CachedFile cachedFile : cachedFiles) {
            cachedFile.deleteWhenPossible();
        }
    }

    private CachedFile newCachedFile() {
        File file = new File(tmpDir, filenameCounter.incrementAndGet() + EXT);
        return new CachedFile(file);
    }

    private void onEviction(CachedFile node) {
        node.deleteWhenPossible();
    }

}
