package com.atlassian.plugin.cache.filecache.impl;

import com.atlassian.plugin.cache.filecache.FileCacheStreamProvider;
import com.atlassian.plugin.servlet.DownloadException;
import org.apache.commons.io.IOUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.BufferedOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;

/**
 * Caches the contents of a stream, and deletes the cache when asked to do so.
 * This class encapsulates the lifecycle of a cached file, from as-yet-uncached, to cached, to needs-deletion, to deleted.
 * <p/>
 * <p>Calls to stream() always succeed, even if the cache was deleted.
 * The initial call to stream() will block other callers to stream() and deleteWhenPossible().
 * <p/>
 * Subsequent calls to stream() don't block other calls.
 * <p/>
 * Subsequent calls to deleteWhenPossible() don't block.
 * <p/>
 * <p>A call to deleteWhenPossible() always results in the file being (eventually) deleted.
 * <p/>
 * <p>Both stream() and deleteWhenPossible() can be called multiple times, in any order, with any
 * level of concurrency.
 *
 * @since v2.13
 */
public class CachedFile
{
    static enum State
    {
        UNCACHED, CACHED, NEEDSDELETE, DELETED
    }

    interface FileOpener
    {
        OutputStream open(File file) throws IOException;
    }

    static class FileInputStreamOpener implements FileOpener
    {
        @Override
        public OutputStream open(File file) throws IOException
        {
            return new FileOutputStream(file);
        }
    }

    private static final Logger log = LoggerFactory.getLogger(CachedFile.class);

    private final File tmpFile;
    private final FileOpener fileOpener;

    /**
     * guards read/writes to `concurrentCount` and `state`
     */
    private final Object lock = new Object();
    private int concurrentCount;
    private State state = State.UNCACHED;

    public CachedFile(File tmpFile)
    {
        this(tmpFile, new FileInputStreamOpener());
    }

    CachedFile(File tmpFile, final FileOpener fileOpener)
    {
        this.tmpFile = tmpFile;
        this.fileOpener = fileOpener;
    }

    public void stream(OutputStream dest, FileCacheStreamProvider input) throws DownloadException
    {

        boolean fromCache = doEnter(input);
        try
        {
            if (fromCache)
            {
                streamFromCache(dest);
            }
            else
            {
                input.writeStream(dest);
            }
        }
        finally
        {
            doExit();
        }
    }

    State getState()
    {
        return state;
    }

    /**
     * Increments the concurrent count, and returns if it is okay
     * to read from the cached file, or not.
     * If necessary, it will create the cached file first.
     * doExit() must be called iff this method returns normally
     */
    private boolean doEnter(FileCacheStreamProvider input)
    {
        synchronized (lock)
        {
            boolean useCache = false;
            if (state == State.UNCACHED)
            {
                try
                {
                    streamToCache(input);
                    state = State.CACHED;
                    useCache = true;
                }
                catch (Exception e)
                {
                    log.warn("Problem caching to disk, skipping cache for this entry", e);
                    state = State.DELETED; // don't bother caching at all
                }
            }
            else if (state == State.CACHED)
            {
                useCache = true;
            }
            else if (state == State.NEEDSDELETE)
            {
                useCache = true; // need to delete, but allow concurrent callers to use it while it's there
            }

            concurrentCount++;
            return useCache && tmpFile.isFile(); // don't use the cache if the file disappeared
        }
    }

    /**
     * Decrements the concurrent count. Will delete the cache if requested, and there are no
     * other concurrent users.
     */
    private void doExit()
    {
        synchronized (lock)
        {
            concurrentCount--;
            if (state == State.NEEDSDELETE && concurrentCount == 0)
            {
                tmpFile.delete();
                state = State.DELETED;
            }
        }
    }

    public void deleteWhenPossible()
    {
        synchronized (lock)
        {
            if (state == State.UNCACHED)
            {
                state = State.DELETED;
            }
            else if (state == State.CACHED)
            {
                state = State.NEEDSDELETE;
            }

            if (state == State.NEEDSDELETE && concurrentCount == 0)
            {
                tmpFile.delete();
                state = State.DELETED;
            }
        }
    }

    private void streamToCache(FileCacheStreamProvider input) throws IOException, DownloadException
    {
        OutputStream cacheout = null;
        try
        {
            cacheout = new BufferedOutputStream(fileOpener.open(tmpFile));
            input.writeStream(cacheout);
        }
        finally
        {
            if (cacheout != null)
            {
                cacheout.close(); // NB: if this fails, we don't want to silently ignore it
            }
        }
    }

    private void streamFromCache(OutputStream out) throws DownloadException
    {
        InputStream in = null;
        try
        {
            in = new FileInputStream(tmpFile);
            IOUtils.copyLarge(in, out);
            out.flush();
        }
        catch (IOException e)
        {
            throw new DownloadException(e);
        }
        finally
        {
            IOUtils.closeQuietly(in);
        }
    }

}