package com.atlassian.plugin.webresource;

import com.atlassian.plugin.PluginAccessor;
import com.atlassian.plugin.elements.ResourceLocation;
import com.atlassian.plugin.servlet.DownloadableResource;
import com.atlassian.plugin.webresource.transformer.WebResourceTransformerModuleDescriptor;
import org.apache.commons.lang.Validate;
import org.dom4j.Element;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.Collections;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

/**
 * Represents a set of transformer invocations for a specific web resource set and extension.  Transformers are retrieved
 * from the plugin system on request, not plugin initialisation, since plugin start order is indeterminate.
 *
 * @since 2.5.0
 */
public class WebResourceTransformation
{
    private final String extension;
    private final Map<String, Element> transformerElements;
    private Logger log = LoggerFactory.getLogger(WebResourceTransformation.class);

    public WebResourceTransformation(Element element)
    {
        Validate.notNull(element.attribute("extension"));
        
        this.extension = "." + element.attributeValue("extension");

        LinkedHashMap<String, Element> transformers = new LinkedHashMap<String, Element>();
        for (Element transformElement : (List<Element>)element.elements("transformer"))
        {
            transformers.put(transformElement.attributeValue("key"), transformElement);
        }
        transformerElements = Collections.unmodifiableMap(transformers);
    }

    public boolean matches(ResourceLocation location)
    {
        String loc = location.getLocation();
        if (loc == null || "".equals(loc.trim()))
        {
            loc = location.getName();
        }
        return loc.endsWith(extension);
    }

    public DownloadableResource transformDownloadableResource(PluginAccessor pluginAccessor, DownloadableResource resource, ResourceLocation resourceLocation, String filePath)
    {
        DownloadableResource lastResource = resource;
        for (Map.Entry<String, Element> entry : transformerElements.entrySet())
        {
            boolean found = false;
            for (WebResourceTransformerModuleDescriptor descriptor : pluginAccessor.getEnabledModuleDescriptorsByClass(WebResourceTransformerModuleDescriptor.class))
            {
                if (descriptor.getKey().equals(entry.getKey()))
                {
                    found = true;
                    lastResource = descriptor.getModule().transform(entry.getValue(), resourceLocation, filePath, lastResource);
                }
            }
            if (!found)
            {
                log.warn("Web resource transformer {} not found for resource {}, skipping", entry.getKey(), resourceLocation.getName());
            }
        }
        return lastResource;
    }

}
