package com.atlassian.plugin.webresource;

import com.atlassian.util.concurrent.LazyReference;
import com.google.common.base.Function;
import com.google.common.base.Supplier;
import com.google.common.base.Suppliers;
import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheLoader;

import static com.google.common.base.Preconditions.checkNotNull;

/**
 * The set of identifying information for a resource.
 */
interface ResourceKey
{
    /** @return the complete key of the module containing the resource. */
    String key();
    /** @return the name of the resource, including the file type suffix. */
    String name();
    /** @return the file type suffix. */
    String suffix();

    /**
     * A ResourceKey, for when we know ahead of time what all the properties are.
     */
    class Strict implements ResourceKey
    {
        final String key;
        final String name;
        final String suffix;

        Strict(String key, String name, String suffix)
        {
            this.key = checkNotNull(key);
            this.name = checkNotNull(name);
            this.suffix = checkNotNull(suffix);
        }

        public String key()
        {
            return key;
        }

        public String name()
        {
            return name;
        }

        public String suffix()
        {
            return suffix;
        }
        @Override
        public boolean equals(Object o)
        {
            if (!(o instanceof ResourceKey))
                return false;
            return Equality.equal(this, (ResourceKey) o);
        }

        @Override
        public int hashCode()
        {

            return Equality.hash(this);
        }
    }

    /**
     * A resource key, for when one or more of the properties are derived.
     */
    class Lazy implements ResourceKey
    {
        final Supplier<String> key;
        final Supplier<String> name;
        final Supplier<String> suffix;

        Lazy(Supplier<String> key, Supplier<String> name, Supplier<String> suffix)
        {
            this.key = checkNotNull(key);
            this.name = checkNotNull(name);
            this.suffix = checkNotNull(suffix);
        }

        public String key()
        {
            return key.get();
        }

        public String name()
        {
            return name.get();
        }

        public String suffix()
        {
            return suffix.get();
        }

        @Override
        public boolean equals(Object o)
        {
            if (!(o instanceof ResourceKey))
                return false;
            return Equality.equal(this, (ResourceKey) o);
        }

        @Override
        public int hashCode()
        {
            return Equality.hash(this);
        }
    }

    /**
     * Builds ResourceKeys.
     */
    class Builder
    {
        /**
         * Builds a lazy key that will create the resourceName when it is actually required
         */
        static ResourceKey lazy(final String completeKey, final Supplier<String> suffix)
        {
            return new ResourceKey.Lazy(
                Suppliers.ofInstance(completeKey),
                new LazyReference<String>()
                {
                    @Override
                    protected String create()
                    {
                        return completeKey + "." + suffix.get();
                    }
                },
                suffix
            );
        }

        static final String DEFAULT_RESOURCE_NAME_PREFIX = "batch";

        static final Function<String, ResourceKey> BATCH_RESOURCE_KEY_CACHE = CacheBuilder.newBuilder().build(
            CacheLoader.from(new Function<String, ResourceKey>()
            {
                @Override
                public ResourceKey apply(String suffix)
                {
                    return new ResourceKey.Strict(DEFAULT_RESOURCE_NAME_PREFIX, DEFAULT_RESOURCE_NAME_PREFIX + "." + suffix, suffix);
                }
            }));

        /**
         * Builds a lazy key that will have a resourceName of "batch.[suffix]".
         * @param suffix
         * @return
         */
        static ResourceKey batch(String suffix)
        {
            // PERF: cache these because we would otherwise create hundreds of them, and they're always the same.
            return BATCH_RESOURCE_KEY_CACHE.apply(suffix);
        }
    }

    /**
     * Equals and hashcode for any ResourceKey.
     */
    class Equality
    {
        static boolean equal(ResourceKey left, ResourceKey right)
        {
            if (!left.key().equals(right.key()))
            {
                return false;
            }
            if (!left.name().equals(right.name()))
            {
                return false;
            }
            return left.suffix().equals(right.suffix());
        }

        public static int hash(ResourceKey resource)
        {
            int result = resource.key().hashCode();
            result = 31 * result + resource.name().hashCode();
            result = 31 * result + resource.suffix().hashCode();
            return result;
        }
    }
}
