package com.atlassian.plugin.servlet.cache.filter;

import com.atlassian.plugin.servlet.cache.model.CacheInformation;
import com.atlassian.plugin.servlet.cache.model.CacheableResponse;
import com.atlassian.plugin.servlet.cache.model.ETagToken;

import javax.servlet.Filter;
import javax.servlet.FilterChain;
import javax.servlet.FilterConfig;
import javax.servlet.ServletException;
import javax.servlet.ServletRequest;
import javax.servlet.ServletResponse;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.io.IOException;
import java.util.regex.Pattern;

import static com.atlassian.plugin.servlet.util.function.FailableConsumer.wrapper;
import static com.google.common.net.HttpHeaders.ETAG;
import static javax.servlet.http.HttpServletResponse.SC_NOT_MODIFIED;

/**
 * <p>Filter responsible for performing caching of resources that have their extension matching {@link ETagCachingFilter#CACHEABLE_FILE_PATTERN}.</p>
 *
 * <p>This filter will cover the following scenarios:</p>
 *
 * <ul>
 *     <li>WHEN If-None-Match matches the ETag generated from a Response body.</li>
 *     <li>THEN set the ETag header with the current ETag and set Http Status code to 304.</li>
 *     <li>AND an empty response body.</li>
 * </ul>
 * <ul>
 *     <li>WHEN If-None-Match does not match the ETag generated from a response body.</li>
 *     <li>THEN set the ETag header with the current ETag.</li>
 *     <li>AND set the response body with the current content.</li>
 * </ul>
 *
 * @since 4.1.19
 */
public class ETagCachingFilter implements Filter {

    private static final Pattern CACHEABLE_FILE_PATTERN = Pattern.compile(".+\\.(css|js)");

    /**
     * Verifies whether the current request is cacheable or not, based on the {@link ETagCachingFilter#CACHEABLE_FILE_PATTERN}.
     *
     * @param request The current request containing the url to be analyzed.
     * @return {@code true}: It should be cached. <br/>
     * {@code false}: It should not be cached.
     */
    private static boolean isCacheableRequest(final ServletRequest request, final ServletResponse response) {
        if (request instanceof HttpServletRequest && response instanceof HttpServletResponse) {
            final HttpServletRequest httpServletRequest = (HttpServletRequest) request;
            return CACHEABLE_FILE_PATTERN
                    .matcher(httpServletRequest.getRequestURI())
                    .matches();
        }
        return false;
    }

    @Override
    public void init(final FilterConfig filterConfig) {
        //not used
    }

    @Override
    public void doFilter(final ServletRequest request,
                         final ServletResponse response,
                         final FilterChain chain) throws IOException, ServletException {
        if (isCacheableRequest(request, response)) {
            final HttpServletRequest httpServletRequest = (HttpServletRequest) request;
            final HttpServletResponse httpServletResponse = (HttpServletResponse) response;
            doFilterWithCachingHeader(httpServletRequest, httpServletResponse, chain);
            return;
        }
        chain.doFilter(request, response);
    }

    private void doFilterWithCachingHeader(final HttpServletRequest request,
                                           final HttpServletResponse response,
                                           final FilterChain chain) throws IOException, ServletException {
        final CacheableResponse wrappedResponse = new CacheableResponse(response);

        chain.doFilter(request, wrappedResponse);

        final CacheInformation cacheInformation = new CacheInformation(request);

        if (cacheInformation.hasFreshContent(wrappedResponse)) {
            wrappedResponse.getContentBody()
                    .ifPresent(wrapper(contentBody -> {
                        response.setContentLength(contentBody.length);
                        response.getOutputStream().write(contentBody);
                        response.getOutputStream().flush();
                    }));
        }
        if (cacheInformation.hasNotFreshContent(wrappedResponse)) {
            response.setStatus(SC_NOT_MODIFIED);
        }
        wrappedResponse.toETagToken()
                .map(ETagToken::getDoubleQuotedValue)
                .ifPresent(etag -> response.setHeader(ETAG, etag));
    }

    @Override
    public void destroy() {
        //not used
    }
}
