package com.atlassian.plugin.web.analytics;


import com.atlassian.analytics.api.annotations.EventName;
import com.atlassian.annotations.VisibleForTesting;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import java.util.zip.CRC32;

/**
 * An analytic event around measuring how long it takes to evaluate a condition on a web-fragment.
 * <p>
 * Added with the intent to be able to: detect problematic conditions, and justify the need for deferred web-fragments.
 * Part of the PLPI work
 *
 * @see <a href="https://ecosystem.atlassian.net/browse/SPFE-341">https://ecosystem.atlassian.net/browse/SPFE-341</a>
 * @since 5.2.0
 */
@EventName("web.fragment.condition.evaluate.time")
public class ConditionEvaluateTimeEvent {

    private static final int ANALYTIC_EVENT_VERSION = 1;
    private static final String ATLASSIAN_PACKAGE_PREFIX = "com.atlassian";

    private final boolean atlassianPlugin;
    private final long evaluationTimeMilliseconds;
    private final String fragmentLocation;
    private final long pluginKeyResponsible;
    private final boolean successfullyEvaluatedCondition;

    /**
     * @param fragmentLocation                The location where the web-fragment is rendered
     * @param pluginKeyResponsible            The plugin where the web-fragment condition comes from
     * @param successfullyEvaluatedCondition  If no exception was thrown during evaluation
     * @param timeAtStartOfEvaluationInMillis The instant immediately before evaluating the condition (used to
     *                                        determine total evaluation time)
     * @param timeAtEndOfEvaluationInMillis   The instant immediately after evaluating the condition (used to
     *                                        determine total evaluation time)
     */
    public ConditionEvaluateTimeEvent(
            @Nullable final String fragmentLocation,
            @Nonnull final String pluginKeyResponsible,
            final boolean successfullyEvaluatedCondition,
            final long timeAtStartOfEvaluationInMillis,
            final long timeAtEndOfEvaluationInMillis
    ) {
        this.atlassianPlugin = isAtlassianPlugin(pluginKeyResponsible);
        this.fragmentLocation = fragmentLocation;
        this.pluginKeyResponsible = hashPluginKeyString(pluginKeyResponsible);
        this.successfullyEvaluatedCondition = successfullyEvaluatedCondition;
        this.evaluationTimeMilliseconds = timeAtEndOfEvaluationInMillis - timeAtStartOfEvaluationInMillis;
    }

    /**
     * A fast, insecure hashing method for the sole purpose of obscuring plugin keys.
     * <p>
     * NOTE: NOT to be used for cryptography
     * NOTE: Intended for usage on the order of 10^4 unique values, should be fine till 10^17 after which the probably
     * of encountering the birthday problem becomes very real.
     *
     * @param pluginKey plugin key to hash
     * @return hashed String of plugin key
     */
    @VisibleForTesting
    static long hashPluginKeyString(@Nonnull String pluginKey) {
        // This is a deliberate choice of a hashing algorithm; it needs to be as fast as possible regardless of CPU
        // architecture (i.e hardware acceleration) since it will be hit on every WebPanel render.
        final CRC32 crc32 = new CRC32(); // Not a static field since it is not thread-safe.
        crc32.update(pluginKey.getBytes(), 0, pluginKey.length()); // CRC32#update(byte[] b) is removed in JDK 9+
        return crc32.getValue();
    }

    @VisibleForTesting
    static boolean isAtlassianPlugin(@Nonnull String pluginKey) {
        // A bit faster than regex since it will be hit on every WebPanel render.
        return pluginKey.startsWith(ATLASSIAN_PACKAGE_PREFIX);
    }

    public int getAnalyticEventVersion() {
        return ANALYTIC_EVENT_VERSION;
    }

    public boolean getAtlassianPlugin() {
        return atlassianPlugin;
    }

    public long getPluginKeyResponsible() {
        return pluginKeyResponsible;
    }

    public boolean getSuccessfullyEvaluatedCondition() {
        return successfullyEvaluatedCondition;
    }

    public long getEvaluationTimeMilliseconds() {
        return evaluationTimeMilliseconds;
    }

    public String getFragmentLocation() {
        return fragmentLocation;
    }
}
