package com.atlassian.plugin.osgi.factory;

import com.atlassian.plugin.ModuleDescriptor;
import com.atlassian.plugin.PluginState;
import com.atlassian.plugin.descriptors.UnrecognisedModuleDescriptor;
import com.atlassian.plugin.event.PluginEventManager;
import com.atlassian.plugin.event.events.PluginModuleAvailableEvent;
import com.atlassian.plugin.event.events.PluginModuleUnavailableEvent;
import com.atlassian.plugin.module.Element;
import com.atlassian.plugin.osgi.external.ListableModuleDescriptorFactory;
import org.osgi.framework.Bundle;
import org.osgi.framework.ServiceReference;
import org.osgi.util.tracker.ServiceTrackerCustomizer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.ArrayList;
import java.util.List;

import static com.google.common.base.Preconditions.checkNotNull;

/**
 * Service tracker that tracks {@link com.atlassian.plugin.osgi.external.ListableModuleDescriptorFactory} instances and handles transforming
 * {@link com.atlassian.plugin.descriptors.UnrecognisedModuleDescriptor}} instances into modules if the new factory supports them. Updates to factories
 * and removal are also handled.
 *
 * @since 2.1.2
 */
class UnrecognizedModuleDescriptorServiceTrackerCustomizer implements ServiceTrackerCustomizer {
    private static final Logger log = LoggerFactory.getLogger(UnrecognizedModuleDescriptorServiceTrackerCustomizer.class);

    private final Bundle bundle;
    private final OsgiPlugin plugin;
    private final PluginEventManager pluginEventManager;

    public UnrecognizedModuleDescriptorServiceTrackerCustomizer(OsgiPlugin plugin, PluginEventManager pluginEventManager) {
        this.plugin = checkNotNull(plugin);
        this.bundle = checkNotNull(plugin.getBundle());
        this.pluginEventManager = checkNotNull(pluginEventManager);
    }

    /**
     * Turns any {@link com.atlassian.plugin.descriptors.UnrecognisedModuleDescriptor} modules that can be handled by the new factory into real
     * modules
     */
    public Object addingService(final ServiceReference serviceReference) {
        final ListableModuleDescriptorFactory factory = (ListableModuleDescriptorFactory) bundle.getBundleContext().getService(serviceReference);

        // Only register the factory if it is or should be being used by this plugin. We still care if they are currently
        // in use because we need to change them to unrecognized descriptors if the factory goes away.
        if (canFactoryResolveUnrecognizedDescriptor(factory) || isFactoryInUse(factory)) {
            return factory;
        } else {
            // The docs seem to indicate returning null is enough to untrack a service, but the source code and tests
            // show otherwise.
            bundle.getBundleContext().ungetService(serviceReference);
            return null;
        }
    }

    /**
     * See if the descriptor factory can resolve any unrecognized descriptors for this plugin, and if so, resolve them
     *
     * @param factory The new module descriptor factory
     * @return True if any were resolved, false otherwise
     */
    private boolean canFactoryResolveUnrecognizedDescriptor(ListableModuleDescriptorFactory factory) {
        boolean usedFactory = false;
        for (final UnrecognisedModuleDescriptor unrecognised : getModuleDescriptorsByDescriptorClass(UnrecognisedModuleDescriptor.class)) {
            final Element source = plugin.getModuleElements().get(unrecognised.getKey());
            if ((source != null) && factory.hasModuleDescriptor(source.getName())) {
                usedFactory = true;
                try {
                    final ModuleDescriptor<?> descriptor = factory.getModuleDescriptor(source.getName());
                    descriptor.init(unrecognised.getPlugin(), source);
                    plugin.addModuleDescriptor(descriptor);
                    log.info("Turned unrecognized plugin module {} into module {}", descriptor.getCompleteKey(), descriptor);
                    pluginEventManager.broadcast(new PluginModuleAvailableEvent(descriptor));
                } catch (final Exception e) {
                    log.error("Unable to transform {} into actual plugin module using factory {}",
                            unrecognised.getCompleteKey(), factory, e);
                    unrecognised.setErrorText(e.getMessage());
                }
            }
        }
        return usedFactory;
    }

    /**
     * Determine if the module descriptor factory is being used by any of the recognized descriptors.
     *
     * @param factory The new descriptor factory
     * @return True if in use, false otherwise
     */
    private boolean isFactoryInUse(ListableModuleDescriptorFactory factory) {
        for (ModuleDescriptor<?> descriptor : plugin.getModuleDescriptors()) {
            for (Class<? extends ModuleDescriptor> descriptorClass : factory.getModuleDescriptorClasses()) {
                if (descriptorClass == descriptor.getClass()) {
                    return true;
                }
            }
        }
        return false;
    }

    public void modifiedService(final ServiceReference serviceReference, final Object o) {
        // do nothing as it is only modifying the properties, which we largely ignore
    }

    /**
     * Reverts any current module descriptors that were provided from the factory being removed into {@link
     * UnrecognisedModuleDescriptor} instances.
     */
    public void removedService(final ServiceReference serviceReference, final Object o) {
        final ListableModuleDescriptorFactory factory = (ListableModuleDescriptorFactory) o;
        for (final Class<? extends ModuleDescriptor> moduleDescriptorClass : factory.getModuleDescriptorClasses()) {
            for (final ModuleDescriptor<?> descriptor : getModuleDescriptorsByDescriptorClass(moduleDescriptorClass)) {
                if (plugin.getPluginState() == PluginState.ENABLED) {
                    pluginEventManager.broadcast(new PluginModuleUnavailableEvent(descriptor));
                    log.info("Removed plugin module {} as its factory was uninstalled", descriptor.getCompleteKey());
                }
                plugin.clearModuleDescriptor(descriptor.getKey());

                if (plugin.isFrameworkShuttingDown()) {
                    continue;
                }
                final UnrecognisedModuleDescriptor unrecognisedModuleDescriptor = new UnrecognisedModuleDescriptor();
                final Element source = plugin.getModuleElements().get(descriptor.getKey());
                if (source != null) {
                    unrecognisedModuleDescriptor.init(plugin, source);
                    unrecognisedModuleDescriptor.setErrorText(UnrecognisedModuleDescriptorFallbackFactory.DESCRIPTOR_TEXT);
                    plugin.addModuleDescriptor(unrecognisedModuleDescriptor);

                    if (plugin.getPluginState() == PluginState.ENABLED) {
                        pluginEventManager.broadcast(new PluginModuleAvailableEvent(unrecognisedModuleDescriptor));
                    }
                }
            }
        }
    }

    <T extends ModuleDescriptor<?>> List<T> getModuleDescriptorsByDescriptorClass(final Class<T> descriptor) {
        final List<T> result = new ArrayList<>();

        for (final ModuleDescriptor<?> moduleDescriptor : plugin.getModuleDescriptors()) {
            if (descriptor.isAssignableFrom(moduleDescriptor.getClass())) {
                result.add(descriptor.cast(moduleDescriptor));
            }
        }
        return result;
    }
}