package com.atlassian.plugin.osgi.hook;

import java.util.Collection;
import java.util.Iterator;
import java.util.Set;

import org.osgi.framework.Bundle;
import org.osgi.framework.hooks.resolver.ResolverHook;
import org.osgi.framework.wiring.BundleCapability;
import org.osgi.framework.wiring.BundleRequirement;
import org.osgi.framework.wiring.BundleRevision;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class DmzResolverHook implements ResolverHook {

    private static final Logger LOG = LoggerFactory.getLogger(DmzResolverHook.class);

    static final String ATTR_WIRING_PACKAGE = "osgi.wiring.package";
    private final InternalPluginDetector pluginDetector;

    public DmzResolverHook(
            Set<String> bundledPluginKeys, Set<String> publicPackages, Set<String> publicPackagesExcludes) {
        this.pluginDetector = new InternalPluginDetector(bundledPluginKeys, publicPackages, publicPackagesExcludes);
    }

    @Override
    public void filterMatches(BundleRequirement requirement, Collection<BundleCapability> possibleExports) {
        Bundle importingPlugin = requirement.getRevision().getBundle();
        boolean isExternalPlugin = !pluginDetector.isInternalPlugin(importingPlugin);

        // this is not only Atlassian Plugin, for example PluginKey:
        // org.eclipse.gemini.blueprint.core
        // org.apache.servicemix.bundles.spring-core
        String pluginKey = pluginDetector.getPluginKeyOrSymbolicName(importingPlugin);

        if (isExternalPlugin) {
            // if external plugin than go into modification
            LOG.debug("Plugin is external, will try to modify the possible exports. PluginKey: {}", pluginKey);
            Iterator<BundleCapability> possibleExportsItr = possibleExports.iterator();

            // iterate over all exports
            while (possibleExportsItr.hasNext()) {
                BundleCapability bundleCapability = possibleExportsItr.next();
                Object bundleCapabilityPackage =
                        bundleCapability.getAttributes().get(ATTR_WIRING_PACKAGE);

                // if internal capability than remove this package export
                if (pluginDetector.isInternalCapability(bundleCapability)) {
                    possibleExportsItr.remove();
                    LOG.warn(
                            "Removed candidate from possible exports. PluginKey: {}. Removed candidate: {}",
                            pluginKey,
                            bundleCapabilityPackage);
                } else {
                    LOG.debug(
                            "Not internal capability. PluginKey: {}. Candidate: {}",
                            pluginKey,
                            bundleCapabilityPackage);
                }
            }
        } else {
            LOG.debug("Plugin is internal, won't be modifying the possible exports. PluginKey: {}", pluginKey);
        }
    }

    @Override
    public void end() {}

    @Override
    public void filterResolvable(Collection<BundleRevision> candidates) {}

    @Override
    public void filterSingletonCollisions(BundleCapability singleton, Collection<BundleCapability> collisions) {}
}
