package com.atlassian.plugins.osgi.test;

import java.io.IOException;
import java.lang.annotation.Annotation;
import java.util.List;
import java.util.Map;

import javax.ws.rs.core.MediaType;

import com.atlassian.plugins.osgi.test.rest.AnnotationTypeAdapter;
import com.atlassian.plugins.osgi.test.rest.TestResultDetailRepresentation;

import com.google.gson.Gson;
import com.google.gson.GsonBuilder;

import org.apache.wink.client.ClientConfig;
import org.apache.wink.client.Resource;
import org.apache.wink.client.RestClient;
import org.junit.AfterClass;
import org.junit.BeforeClass;
import org.junit.internal.AssumptionViolatedException;
import org.junit.internal.runners.model.EachTestNotifier;
import org.junit.internal.runners.statements.RunAfters;
import org.junit.internal.runners.statements.RunBefores;
import org.junit.rules.RunRules;
import org.junit.rules.TestRule;
import org.junit.runner.Description;
import org.junit.runner.Result;
import org.junit.runner.notification.Failure;
import org.junit.runner.notification.RunNotifier;
import org.junit.runner.notification.StoppedByUserException;
import org.junit.runners.BlockJUnit4ClassRunner;
import org.junit.runners.model.FrameworkMethod;
import org.junit.runners.model.InitializationError;
import org.junit.runners.model.Statement;

import static org.junit.internal.runners.rules.RuleFieldValidator.CLASS_RULE_VALIDATOR;
import static org.junit.internal.runners.rules.RuleFieldValidator.RULE_VALIDATOR;

/**
 * @since version
 */
public class AtlassianPluginsTestRunner extends BlockJUnit4ClassRunner
{

    /**
     * Creates a BlockJUnit4ClassRunner to run {@code klass}
     *
     * @throws org.junit.runners.model.InitializationError
     *          if the test class is malformed.
     */
    public AtlassianPluginsTestRunner(Class<?> klass) throws InitializationError
    {
        super(klass);
    }

    @Override
    public void run(final RunNotifier notifier)
    {
        EachTestNotifier testNotifier = new EachTestNotifier(notifier, getDescription());
        try
        {
            String[] packageParts = getTestClass().getJavaClass().getPackage().getName().split("\\.");
            
            if(null == packageParts || packageParts.length < 1 || !packageParts[0].equals("it"))
            {
                throw new Exception("the class [" + getTestClass().getJavaClass().getName() + "] is annotated with @RunWith(AtlassianPluginsTestRunner.class) but it is not in the 'it.' package." +
                        "\nPlease move the class into the 'it.' package or remove the @RunWith annotation");    
            }
            
            runViaRestCall(getDescription(), notifier);
        }
        catch (AssumptionViolatedException e)
        {
            testNotifier.fireTestIgnored();
        }
        catch (StoppedByUserException e)
        {
            throw e;
        }
        catch (Throwable e)
        {
            testNotifier.addFailure(e);
        }
    }

    private void runViaRestCall(Description description, RunNotifier notifier) throws IOException
    {

        String resourceUrl = System.getProperty("baseurl") + "/rest/atlassiantestrunner/1.0/runtest/" + description.getClassName();

        ClientConfig config = new ClientConfig();
        config.readTimeout(1800000);

        RestClient client = new RestClient(config);

        Resource resource = client.resource(resourceUrl);

        String response = resource.accept(MediaType.APPLICATION_JSON).get(String.class);

        GsonBuilder gsonBuilder = new GsonBuilder();
        gsonBuilder.registerTypeAdapter(Annotation.class, new AnnotationTypeAdapter());

        Gson gson = gsonBuilder.create();
        
        TestResultDetailRepresentation osgiResult = gson.fromJson(response,TestResultDetailRepresentation.class);
        
        for(String pMethodName : osgiResult.getPassedMethods())
        {
            EachTestNotifier testNotifier = new EachTestNotifier(notifier, Description.createTestDescription(getTestClass().getJavaClass(),pMethodName));
            testNotifier.fireTestStarted();
            testNotifier.fireTestFinished();
        }

        for(String iMethodName : osgiResult.getIgnoredMethods())
        {
            EachTestNotifier testNotifier = new EachTestNotifier(notifier, Description.createTestDescription(getTestClass().getJavaClass(),iMethodName));
            testNotifier.fireTestIgnored();
        }

        for(Map.Entry<String,Failure> entry : osgiResult.getFailedMethods().entrySet())
        {
            String fMethodName = entry.getKey();
            Failure f = entry.getValue();
            EachTestNotifier testNotifier = new EachTestNotifier(notifier, Description.createTestDescription(getTestClass().getJavaClass(),fMethodName));
            testNotifier.addFailure(f.getException());
        }
    }

    @Override
    protected void collectInitializationErrors(List<Throwable> errors)
    {
        validatePublicVoidNoArgMethods(BeforeClass.class, false, errors);
        validatePublicVoidNoArgMethods(AfterClass.class, false, errors);
        CLASS_RULE_VALIDATOR.validate(getTestClass(), errors);
        validateInstanceMethods(errors);
        RULE_VALIDATOR.validate(getTestClass(), errors);
    }

}
