package com.atlassian.plugins.osgi.javaconfig;

import com.atlassian.annotations.PublicApi;
import com.atlassian.annotations.nullability.ParametersAreNonnullByDefault;
import com.atlassian.annotations.nullability.ReturnValuesAreNonnullByDefault;
import org.eclipse.gemini.blueprint.service.importer.support.CollectionType;

import javax.annotation.Nonnull;
import java.util.Comparator;
import java.util.List;
import java.util.Set;

import static java.util.Objects.requireNonNull;

/**
 * This class defines the type of service collection to import from OSGi. It acts as a type token.
 *
 * @param <T> the type of service collection
 */
@PublicApi
@ParametersAreNonnullByDefault
@ReturnValuesAreNonnullByDefault
public final class ServiceCollection<T> {

    private final Class<?> serviceClass;

    private final CollectionType collectionType;

    private final Comparator<?> comparator;

    private ServiceCollection(Class<?> serviceClass, CollectionType collectionType) {
        this(serviceClass, collectionType, null);
    }

    private ServiceCollection(Class<?> serviceClass, CollectionType collectionType, Comparator<?> comparator) {
        requireNonNull(serviceClass," Service class should not be null.");
        this.serviceClass = serviceClass;
        this.collectionType = collectionType;
        this.comparator = comparator;
    }

    public static <S> ServiceCollection<List<S>> list(@Nonnull Class<S> serviceClass) {
        return new ServiceCollection<>(serviceClass, CollectionType.LIST);
    }

    public static <S> ServiceCollection<List<S>> sortedList(@Nonnull Class<S> serviceClass) {
        return new ServiceCollection<>(serviceClass, CollectionType.SORTED_LIST);
    }

    public static <S> ServiceCollection<List<S>> sortedList(@Nonnull Class<S> serviceClass, Comparator<? super S> comparator) {
        return new ServiceCollection<>(serviceClass, CollectionType.SORTED_LIST, comparator);
    }

    public static <S> ServiceCollection<Set<S>> set(@Nonnull Class<S> serviceClass) {
        return new ServiceCollection<>(serviceClass, CollectionType.SET);
    }

    public static <S> ServiceCollection<Set<S>> sortedSet(@Nonnull Class<S> serviceClass) {
        return new ServiceCollection<>(serviceClass, CollectionType.SORTED_SET);
    }

    public static <S> ServiceCollection<Set<S>> sortedSet(@Nonnull Class<S> serviceClass, Comparator<? super S> comparator) {
        return new ServiceCollection<>(serviceClass, CollectionType.SORTED_SET, comparator);
    }

    public Comparator<?> getComparator() {
        return comparator;
    }

    CollectionType getCollectionType() {
        return collectionType;
    }

    Class<?> getServiceClass() {
        return serviceClass;
    }
}
