package com.atlassian.plugins.osgi.javaconfig;

import com.atlassian.annotations.PublicApi;
import org.eclipse.gemini.blueprint.service.importer.support.Availability;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import java.time.Duration;
import java.util.Objects;

import static java.util.Objects.requireNonNull;
import static org.eclipse.gemini.blueprint.service.importer.support.Availability.MANDATORY;
import static org.eclipse.gemini.blueprint.service.importer.support.Availability.OPTIONAL;

/**
 * The user-provided options for importing a service from the OSGi service registry.
 *
 * @since 0.3
 */
@PublicApi
public class ImportOptions {

    /**
     * Factory method for the default options, namely:
     * <ul>
     *     <li>no LDAP filter</li>
     *     <li>a timeout of five minutes (being the default timeout used by the {@code osgi:reference} XML element)</li>
     *     <li>the import is mandatory</li>
     * </ul>
     */
    public static ImportOptions defaultOptions() {
        return new ImportOptions();
    }

    private Availability availability = MANDATORY;

    private Duration timeout = Duration.ofMinutes(5);

    private String filter;

    private ImportOptions() {}

    /**
     * Makes this service import optional. This is equivalent to assigning a {@code cardinality} of {@code 0..1} in XML.
     *
     * @return this instance
     * @since 0.5
     */
    public ImportOptions optional() {
        this.availability = OPTIONAL;
        return this;
    }

    /**
     * Specifies the <a href="https://osgi.org/javadoc/r2/org/osgi/framework/Filter.html">LDAP filter</a> to apply when
     * looking up the service in OSGi. For example a filter of {@code "(foo=bar)"} means a service will only be imported
     * if its {@code foo} property is set to {@code bar}.
     *
     * @param filter the filter
     * @return this instance
     */
    public ImportOptions withFilter(final String filter) {
        this.filter = filter;
        return this;
    }

    /**
     * Sets the time to wait for the imported service to be available in the OSGi service registry.
     *
     * @param timeout the timeout
     * @return this instance
     */
    public ImportOptions withTimeout(final Duration timeout) {
        this.timeout = requireNonNull(timeout);
        return this;
    }

    /**
     * Returns the availability of the service.
     *
     * @return see description
     * @since 0.5
     */
    @Nonnull
    public Availability getAvailability() {
        return availability;
    }

    /**
     * Returns the LDAP filter, if any.
     *
     * @return see {@link #withFilter(String)}
     */
    @Nullable
    public String getFilter() {
        return filter;
    }

    /**
     * Returns the timeout to wait for the service to be available in OSGi.
     *
     * @return see description
     */
    public Duration getTimeout() {
        return timeout;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (o == null || getClass() != o.getClass()) return false;
        ImportOptions that = (ImportOptions) o;
        return availability == that.availability && Objects.equals(timeout, that.timeout) && Objects.equals(filter, that.filter);
    }

    @Override
    public int hashCode() {
        return Objects.hash(availability, timeout, filter);
    }
}
