package com.atlassian.plugins.osgi.javaconfig.conditions.product;

import com.atlassian.annotations.PublicApi;

/**
 * A Spring {@link org.springframework.context.annotation.Condition} that only returns <code>true</code> if the plugin
 * is running within Jira.
 *
 * Here's some typical usage scenarios:
 *
 * <pre>
 *     // Here the condition is applied at the class level, because the method return type doesn't exist in all
 *     // products (only Jira). Applying the condition at the method level would cause Spring to throw a
 *     // ClassNotFoundException when it attempts to reflect into the declared methods, if the currently running product
 *     // is not Jira.
 *     &#064;Configuration
 *     &#064;Conditional(JiraOnly.class)
 *     public class JiraSpecificBeans {
 *
 *         &#064;Bean
 *         public IssueService issueService() {
 *             return OsgiServices.importOsgiService(IssueService.class);
 *         }
 *     }
 *
 *     // Here the condition *can* be applied at the method level, because the IssueService reference in the method
 *     // return type is erased at runtime, meaning that Spring will not attempt to load the IssueService class.
 *     &#064;Configuration
 *     public class MyBeans {
 *
 *         &#064;Bean
 *         &#064;Conditional(JiraOnly.class)
 *         public FactoryBean&lt;IssueService&gt; issueServiceFactoryBean() {
 *             return OsgiServices.factoryBeanForOsgiService(FooService.class);
 *         }
 *     }
 * </pre>
 *
 * @since 0.1
 */
@PublicApi
public final class JiraOnly extends AbstractProductCondition {

    /**
     * Constructor invoked reflectively by Spring.
     */
    public JiraOnly() {
        super("com.atlassian.jira.bc.issue.IssueService");
    }
}
