define('atlassian-nps-plugin/js/nps/main', [
    'atlassian-nps-plugin/js/amd-shims/ajs',
    'atlassian-nps-plugin/js/nps/survey-dialog',
    'atlassian-nps-plugin/js/nps/survey-flags',
    'atlassian-nps-plugin/js/nps/client-storage',
    'atlassian-nps-plugin/js/nps/session-manager',
    'atlassian-nps-plugin/js/nps/submission'
],
/**
 * Handles the launching of the NPS survey and binds the different events and behaviour relating to
 * how the user can interact with the survey (closing it, cancelling, etc.)
 * @param AJS
 * @param SurveyDialog modal dialog with the NPS survey
 * @param SurveyFlags notification/confirmation NPS flags
 * @param ClientStorage client-side storage mechanism name spaced by user for retrieving scheduling data
 * @param Session calculates and manages the fine-grained display conditions
 * @param Submission handles the firing of analytics events
 * @returns {object} Session manager implementation
 */
function (
    AJS,
    SurveyDialog,
    SurveyFlags,
    ClientStorage,
    Session,
    Submission
) {
    "use strict";

    function launch(ConfigManager) {
        AJS.bind("/nps/openSurvey", function () {
            Submission.startSurvey();
            ClientStorage.remove("displayCount");
            ConfigManager.scheduleNextSurveyDate().then(function(){
                Session.reset();
                SurveyDialog.showDialog();
            });
        });

        AJS.bind("/nps/cancelSurvey", function (e, context) {
            Submission.cancelSurvey(context);
            if (context === "notification") {
                ClientStorage.remove("displayCount");
                ConfigManager.scheduleNextSurveyDate().then(function(){
                    Session.reset();
                    incrementDismissCount(ConfigManager);
                });
            } else {
                incrementDismissCount(ConfigManager);
            }
        });

        AJS.bind("/nps/dismissSurvey", function (e, context) {
            Submission.cancelSurvey(context);
            ConfigManager.setConfig("optedOut", true);
            ClientStorage.remove("displayCount");
        });

        AJS.bind("/nps/surveyCompleted", function (e, eventAttrs) {
            Submission.submitSurvey(eventAttrs);
            ConfigManager.setConfig("dismissedCount", 0);
            SurveyFlags.showConfirmationFlag();

            if (AJS.DarkFeatures && AJS.DarkFeatures.isEnabled("nps.survey.inline.dialog")) {
                ClientStorage.remove("displayCount");
                ConfigManager.scheduleNextSurveyDate().then(function(){
                    Session.reset();
                });
            }
        });

        var dismissedCount = parseInt(ConfigManager.getCachedData().dismissedCount, 10);
        var showOptOutOption = dismissedCount >= 3;

        if (AJS.DarkFeatures && AJS.DarkFeatures.isEnabled("nps.survey.inline.dialog")) {
            Submission.startSurvey();
            // It's difficult to declare Dark Feature cross products and try to avoid loading redundant dependency (inline dialog 2)
            require(['atlassian-nps-plugin/js/nps/survey-inline-dialog'], function(SurveyInlineDialog) {
                SurveyInlineDialog.show(showOptOutOption);
            });
        } else {
            SurveyFlags.showNotificationFlag(showOptOutOption);
        }
    }

    function incrementDismissCount(ConfigManager) {
        var dismissedCountInt = parseInt(ConfigManager.getCachedData().dismissedCount, 10);
        ConfigManager.setConfig("dismissedCount", dismissedCountInt + 1);
    }

    return {
        launch: launch
    };
});
