package com.atlassian.plugins.navlink.util.url;

import org.apache.commons.lang.StringUtils;

import javax.annotation.Nonnull;
import javax.servlet.http.HttpServletRequest;

/**
 * <p>Extract the self url from the request. In fact, it is the request URI without the context path.</p>
 *
 * <p><strong>If the request is a forward, then the original request URI is used.</strong></p>
 *
 * @see javax.servlet.http.HttpServletRequest#getContextPath()
 * @see javax.servlet.http.HttpServletRequest#getRequestURI()
 */
public class SelfUrl
{
    static final String FORWARD_REQUEST_URI_ATTRIBUTE_NAME = "javax.servlet.forward.request_uri";

    /**
     * Extract the self url (the request URI without the context path) from the request. If this request was forwared,
     * the original request URI is evaluated. The self URL depends on the actual request URL and reflects when the server
     * is reachable through multiple dns addresses (e.g. the self URL is localhost when the request is
     * http://localhost/resource and example.com is it is http://example.com/resource).
     *
     * @param httpServletRequest the request to be used
     * @return the self url
     *
     * @see SelfUrl
     */
    @Nonnull
    public static String extractFrom(@Nonnull final HttpServletRequest httpServletRequest)
    {
        final String requestUri = getRequestURI(httpServletRequest);
        final String contextPath = httpServletRequest.getContextPath();
        return StringUtils.removeStart(requestUri, contextPath);
    }

    private static String getRequestURI(final HttpServletRequest httpServletRequest)
    {
        final Object attribute = httpServletRequest.getAttribute(FORWARD_REQUEST_URI_ATTRIBUTE_NAME);
        if (attribute != null)
        {
            return attribute.toString();
        }
        else
        {
            return httpServletRequest.getRequestURI();
        }
    }
}
