package com.atlassian.plugins.navlink.util.url;

import com.google.common.base.Strings;
import com.google.common.collect.ImmutableMap;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.annotation.concurrent.Immutable;
import javax.servlet.http.HttpServletRequest;
import java.net.MalformedURLException;
import java.net.URL;
import java.util.Map;

import static com.google.common.base.Preconditions.checkNotNull;

/**
 * The base url denotes the first part of the url starting with the protocol and ending with the context path:
 * <code>scheme://serverName[:port]/[contextPath]</code>. It is mainly used to resolve relative links.
 */
@Immutable
public class BaseUrl
{
    private static final Map<String, Integer> DEFAULT_PORTS = ImmutableMap.of("http", 80, "https", 443);

    private final String baseUrl;

    public BaseUrl(@Nonnull final String baseUrl)
    {
        this.baseUrl = checkNotNull(baseUrl);
    }

    /**
     * Attempts to construct a base url using the {@link javax.servlet.http.HttpServletRequest} object. In contrast to
     * the self URL, the base URL depends on the server name and not on the actual requested resource. So despite the
     * request is http://localhost/resource, the server name can be example.com and thus result in the base ULR
     * http://example.com.
     *
     * @param request The incoming http request
     * @return an instance containing the absolute base URL
     * @throws IllegalArgumentException if the request contained a non-valid url
     */
    @Nonnull
    public static BaseUrl extractFrom(final HttpServletRequest request)
    {
        checkNotNull(request);

        final String scheme = Strings.nullToEmpty(request.getScheme());
        final String serverName = Strings.nullToEmpty(request.getServerName());
        final int port = isDefaultPort(scheme, request.getServerPort()) ? -1 : request.getServerPort();
        final String contextPath = Strings.nullToEmpty(request.getContextPath());

        try
        {
            return createBaseUrl(scheme, serverName, port, contextPath);
        }
        catch (MalformedURLException e)
        {
            throw new IllegalArgumentException("Failed to extract a valid url from the request", e);
        }
    }

    /**
     * Construct a base url by looking up the given system property and falling back to the given default value is the
     * system property is not defined.
     * @param key the system property to be checked first
     * @param def the default value to be used in case the system property is not defined
     * @return an instance containing either the value returned by the system property or the given default value
     */
    public static BaseUrl fromSystemProperty(final String key, @Nullable final String def)
    {
        checkNotNull(key, "key");
        checkNotNull(def, "def");
        return new BaseUrl(System.getProperty(key, def));
    }

    @Nonnull
    public String getBaseUrl()
    {
        return baseUrl;
    }

    /**
     * Construct a full url by prepending the base url to the given parameter. If the url is already resolved - it starts
     * with a http or https protocol - it is returned unchanged.
     *
     * @param path the path to be appended to the base url
     * @return an url of the form: base url + path (unless the url is already resolved)
     */
    public String resolve(@Nullable final String path)
    {
        return UrlFactory.toAbsoluteUrl(baseUrl, Strings.nullToEmpty(path));
    }

    @Nonnull
    private static BaseUrl createBaseUrl(final String scheme, final String serverName, final int port, final String contextPath) throws MalformedURLException
    {
        return new BaseUrl(new URL(scheme, serverName, port, contextPath).toExternalForm());
    }

    private static boolean isDefaultPort(@Nonnull final String scheme, final int port)
    {
        final Integer defaultPort = DEFAULT_PORTS.get(scheme.toLowerCase());
        return defaultPort != null && defaultPort == port;
    }

    @Override
    public boolean equals(Object o)
    {
        if (this == o)
        {
            return true;
        }
        else if (o == null || getClass() != o.getClass())
        {
            return false;
        }
        else
        {
            final BaseUrl other = (BaseUrl) o;
            return baseUrl.equals(other.baseUrl);
        }
    }

    @Override
    public int hashCode()
    {
        return baseUrl.hashCode();
    }

    @Override
    public String toString()
    {
        return "BaseUrl{" + baseUrl + '}';
    }
}
