package com.atlassian.plugins.navlink.util;

import javax.ws.rs.core.CacheControl;

public class CacheControlFactory
{
    private static final int TEN_MINUTES = 600;
    private static final int TWENTY_FOUR_HOURS = 86400;

    public static final int MAX_AGE_IN_SECONDS = Integer.getInteger("navlink.cachecontrol.maxage", TEN_MINUTES);
    public static final int STALE_WHILE_REVALIDATE_IN_SECONDS = Integer.getInteger("navlink.cachecontrol.stalewhilerevalidate", TWENTY_FOUR_HOURS);
    public static final int STALE_IF_ERROR_IN_SECONDS = Integer.getInteger("navlink.cachecontrol.stateiferror", TWENTY_FOUR_HOURS);

    /**
     * <p>This setting will allow the client to cache the document and revalidate it in the background while using the
     * stale copy from the cache.</p>
     *
     * <p>The first cache control, <code>max-age</code>, will specify how long the document can be served from the
     * client's cache without asking the server after all. After that period is expired, the cached document can be
     * served for another period as a stale copy while a background thread tries to revalidate / update it (=
     * <code>stale-while-revalidate</code> cache control extension, may not be supported by all clients). After that
     * second period is expires, the document is invalid and won't be served any longer from the cache.</p>
     *
     * <p>Furthermore, the <code>stale-if-error</code> cache control extension allows the client to use the stale
     * document for a specified time even when the server returns an error and thus the document couldn't be
     * revalidated.</p>
     *
     * @return a cache control with the following cache controls set: max-age, stale-while-revalidate and stale-if-error
     * @see <a href="http://tools.ietf.org/html/rfc5861">RFC 5861 - HTTP Cache-Control Extensions for Stale Content</a>
     */
    public static CacheControl withConfiguredMaxAgeAndStaleContentExtension() {
        final CacheControl cacheControl = new CacheControl();
        cacheControl.setMaxAge(MAX_AGE_IN_SECONDS);
        cacheControl.getCacheExtension().put("stale-while-revalidate", Integer.toString(STALE_WHILE_REVALIDATE_IN_SECONDS));
        cacheControl.getCacheExtension().put("stale-if-error", Integer.toString(STALE_IF_ERROR_IN_SECONDS));
        return cacheControl;
    }

    /**
     * Sets up a CacheControl with no-cache set.
     * @return a cache control with the no-cache set
     */
    public static CacheControl withNoCache() {
        final CacheControl cacheControl = new CacheControl();
        cacheControl.setNoCache(true);
        cacheControl.setNoStore(true);
        return cacheControl;
    }
}
