package com.atlassian.plugins.navlink.producer.navigation.rest;

import com.google.common.base.Strings;
import org.apache.commons.lang.LocaleUtils;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.servlet.http.HttpServletRequest;
import java.util.Locale;

/**
 * Encode and decode the language parameter of the navigation REST endpoint.
 */
public class LanguageParameter
{
    static final String PARAMETER_NAME = "lang";

    @Nonnull
    public static String encodeValue(@Nullable final Locale locale)
    {
        final StringBuilder builder = new StringBuilder();
        if (locale != null)
        {
            builder.append(locale.getLanguage());
            if (!Strings.isNullOrEmpty(locale.getCountry()))
            {
                builder.append("-").append(locale.getCountry());
            }
        }
        return builder.toString();
    }

    @Nonnull
    public static Locale extractFrom(@Nonnull final HttpServletRequest httpServletRequest, @Nonnull final Locale defaultValue)
    {
        final String parameterValue = extractParameterValue(httpServletRequest);
        final Locale locale = isValid(parameterValue) ? convertToLocale(parameterValue) : null;
        return locale != null ? locale : defaultValue;
    }

    @Nullable
    private static String extractParameterValue(@Nonnull final HttpServletRequest httpServletRequest)
    {
        return httpServletRequest.getParameter(PARAMETER_NAME);
    }

    private static boolean isValid(@Nullable final String parameterValue)
    {
        return !Strings.isNullOrEmpty(parameterValue);
    }

    @Nullable
    private static Locale convertToLocale(@Nullable final String parameterValue)
    {
        final String decodedParameter = decodeValue(parameterValue);

        try
        {
            return LocaleUtils.toLocale(decodedParameter);
        }
        catch (IllegalArgumentException e)
        {
            return null;
        }
    }

    @Nonnull
    private static String decodeValue(@Nullable final String parameterValue)
    {
        return Strings.nullToEmpty(parameterValue).replace("-", "_");
    }
}
