package com.atlassian.plugins.navlink.producer.navigation.links;

import com.atlassian.plugins.navlink.producer.capabilities.services.ApplicationTypeService;
import org.joda.time.DateTime;

import static com.google.common.base.Preconditions.checkNotNull;

/**
 * Base class to use by builders of subclasses of {@link NavigationLinkBase}
 *
 * @since 2.2.3
 */
public abstract class NavigationLinkBuilderBase<B extends NavigationLinkBuilderBase<B, NL>, NL extends NavigationLinkBase>
{

    protected String key;
    protected String href;
    protected String baseUrl;
    protected String iconUrl;
    protected LinkSource source = LinkSource.unknown();
    protected int weight = Integer.MAX_VALUE;
    protected boolean self = false;

    // TODO: see comment in NavigationLinkBase
    protected String applicationType = ApplicationTypeService.DEFAULT_APPLICATION_TYPE;
    protected DateTime buildDate = new DateTime(0);

    private final Class<B> builderClass;

    protected NavigationLinkBuilderBase(Class<B> bClass)
    {
        this.builderClass = checkNotNull(bClass);
    }

    public final B copy(NavigationLinkBase that)
    {
        return key(that.getKey())
                .href(that.getHref())
                .baseUrl(that.getBaseUrl())
                .iconUrl(that.getIconUrl())
                .applicationType(that.getApplicationType())
                .buildDate(that.getBuildDate())
                .source(that.getSource())
                .weight(that.weight())
                .self(that.isSelf());
    }

    public final B key(String key)
    {
        this.key = key;
        return asTargetInstance();
    }

    public final B href(String href)
    {
        this.href = href;
        return asTargetInstance();
    }

    public final B baseUrl(String baseUrl)
    {
        this.baseUrl = baseUrl;
        return asTargetInstance();
    }

    public final B iconUrl(String iconUrl)
    {
        this.iconUrl = iconUrl;
        return asTargetInstance();
    }

    public final B applicationType(String applicationType)
    {
        this.applicationType = applicationType;
        return asTargetInstance();
    }

    public final B source(LinkSource source)
    {
        this.source = source;
        return asTargetInstance();
    }

    public final B weight(int weight)
    {
        this.weight = weight;
        return asTargetInstance();
    }

    public final B self(boolean self)
    {
        this.self = self;
        return asTargetInstance();
    }

    public final B buildDate(DateTime buildDate)
    {
        this.buildDate = buildDate;
        return asTargetInstance();
    }


    protected final B asTargetInstance()
    {
        return builderClass.cast(this);
    }
}
