package com.atlassian.plugins.navlink.producer.navigation.links;

import com.google.common.base.Objects;

/**
 * <p/>
 * Represents a source application of the link - the link may come from the local, or remote application.
 *
 * <p/>
 * For unknown source, use the
 *
 * @since 3,0
 */
public abstract class LinkSource
{

    public static LinkSource local(String id)
    {
        return new Local(id);
    }

    public static LinkSource localDefault()
    {
        return Local.DEFAULT;
    }

    public static LinkSource remote(String id)
    {
        return new Remote(id);
    }

    public static LinkSource remoteDefault()
    {
        return Remote.DEFAULT;
    }

    public static LinkSource unknown()
    {
        return Unknown.INSTANCE;
    }

    private final String id;
    private final SourceType type;

    private LinkSource(String id, SourceType type)
    {
        this.id = id;
        this.type = type;
    }

    public final String id()
    {
        return id;
    }

    public final SourceType type()
    {
        return type;
    }

    private static final class Local extends LinkSource
    {
        private static final Local DEFAULT = new Local(null);

        private Local(String id)
        {
            super(id, SourceType.LOCAL);
        }
    }

    private static final class Remote extends LinkSource
    {
        private static final Remote DEFAULT = new Remote(null);

        private Remote(String id)
        {
            super(id, SourceType.REMOTE);
        }
    }

    private static final class Unknown extends LinkSource
    {
        private static final Unknown INSTANCE = new Unknown();

        private Unknown()
        {
            super(null, SourceType.UNKNOWN);
        }
    }

    @Override
    public int hashCode()
    {
        return Objects.hashCode(id, type);
    }

    @Override
    public boolean equals(Object obj)
    {
        if (obj == this)
        {
            return true;
        }
        else if (!(obj instanceof LinkSource))
        {
            return false;
        }
        LinkSource that = (LinkSource) obj;
        return Objects.equal(this.id, that.id) &&
                Objects.equal(this.type, that.type);
    }

    @Override
    public String toString()
    {
        return "LinkSource{" +
                "id='" + id + '\'' +
                ", type=" + type +
                '}';
    }
}
