package com.atlassian.plugins.navlink.producer.contentlinks.services;

import com.atlassian.applinks.api.ApplicationLink;
import com.atlassian.applinks.api.CredentialsRequiredException;
import com.atlassian.applinks.api.EntityType;
import com.atlassian.applinks.spi.application.TypeId;
import com.atlassian.applinks.spi.link.MutatingEntityLinkService;
import com.atlassian.applinks.spi.util.TypeAccessor;
import com.atlassian.fugue.Either;
import com.atlassian.fugue.Option;
import com.atlassian.fugue.Options;
import com.atlassian.fugue.Pair;
import com.atlassian.fugue.Suppliers;
import com.atlassian.plugin.PluginAccessor;
import com.atlassian.plugin.web.descriptors.ConditionalDescriptor;
import com.atlassian.plugin.web.descriptors.WeightedDescriptorComparator;
import com.atlassian.plugins.navlink.consumer.http.UserAgentProperty;
import com.atlassian.plugins.navlink.consumer.menu.services.RemoteApplications;
import com.atlassian.plugins.navlink.consumer.projectshortcuts.rest.UnauthenticatedRemoteApplication;
import com.atlassian.plugins.navlink.producer.capabilities.CapabilityKey;
import com.atlassian.plugins.navlink.producer.capabilities.RemoteApplicationWithCapabilities;
import com.atlassian.plugins.navlink.producer.contentlinks.plugin.ContentLinkModuleDescriptor;
import com.atlassian.plugins.navlink.producer.contentlinks.rest.ContentLinkEntity;
import com.atlassian.plugins.navlink.util.executor.DaemonExecutorService;
import com.atlassian.util.concurrent.Nullable;
import com.google.common.base.Function;
import com.google.common.base.Supplier;
import com.google.common.collect.Collections2;
import com.google.common.collect.Iterables;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.annotation.Nonnull;
import java.net.URI;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeUnit;

import static com.atlassian.fugue.Either.left;
import static com.atlassian.fugue.Pair.pair;

public class DefaultContentLinksService implements ContentLinksService
{
    private static final Logger log = LoggerFactory.getLogger(DefaultContentLinksService.class);
    public static final WeightedDescriptorComparator WEIGHTED_DESCRIPTOR_COMPARATOR = new WeightedDescriptorComparator();

    private final PluginAccessor pluginAccessor;
    private final MutatingEntityLinkService mutatingEntityLinkService;
    private final TypeAccessor typeAccessor;
    private final DaemonExecutorService executor;
    private final RemoteApplications remoteApplications;
    private final UserAgentProperty userAgentProperty;

    public DefaultContentLinksService(PluginAccessor pluginAccessor, MutatingEntityLinkService mutatingEntityLinkService,
                                  TypeAccessor typeAccessor, DaemonExecutorService executor, RemoteApplications remoteApplications, UserAgentProperty userAgentProperty)
    {
        this.pluginAccessor = pluginAccessor;
        this.mutatingEntityLinkService = mutatingEntityLinkService;
        this.typeAccessor = typeAccessor;
        this.executor = executor;
        this.remoteApplications = remoteApplications;
        this.userAgentProperty = userAgentProperty;
    }

    @Override
    @Nonnull
    public List<ContentLinkModuleDescriptor> getAllLocalContentLinks(@Nonnull Map<String, Object> context,
            @Nullable final TypeId entityType)
    {
        List<ContentLinkModuleDescriptor> descriptors = pluginAccessor
                .getEnabledModuleDescriptorsByClass(ContentLinkModuleDescriptor.class);
        descriptors = filterFragmentsByCondition(descriptors, context);
        if (entityType != null)
        {
            descriptors = filterFragmentsByTypeId(descriptors, entityType);
        }
        Collections.sort(descriptors, WEIGHTED_DESCRIPTOR_COMPARATOR);
        return descriptors;
    }

    /**
     * Propogates run time exceptions from applinks..
     *
     * @param key
     * @param entityTypeId
     * @return
     */
    @Override
    @Nonnull
    public List<ContentLinkEntity> getAllRemoteContentLinks(@Nonnull String key, @Nonnull final TypeId entityTypeId)
    {
        return new ArrayList<ContentLinkEntity>(executeRemoteContentLinksCollector(key, entityTypeId, new Function<ContentLinkCapability, Callable<Collection<ContentLinkEntity>>>()
        {
            @Override
            public Callable<Collection<ContentLinkEntity>> apply(final ContentLinkCapability contentLinkCapability)
            {
                return new Callable<Collection<ContentLinkEntity>>()
                {
                    public List<ContentLinkEntity> call()
                    {
                        try
                        {
                            ContentLinkClient client = new ContentLinkClient(userAgentProperty);
                            return client.getContentLinks(contentLinkCapability);
                        }
                        catch(Exception ex)
                        {
                            log.error("Could not get project shortcuts for entity {}\n on app {}", contentLinkCapability.getEntityLink(),
                                    contentLinkCapability.getEntityLink().getApplicationLink().getName());
                            log.debug("More details", ex);
                            return Collections.emptyList();
                        }
                    }
                };
            }
        }, Suppliers.<Collection<ContentLinkEntity>>ofInstance(Collections.<ContentLinkEntity>emptyList())));
    }

    /**
     * Retrieves content links from all linked entities, preserving information on Applications which require authentication
     *
     * @param key of the current entity
     * @param entityTypeId of the current entity
     * @return project shortcuts from all linked entities, along with all unauthenticated remote applications.
     */
    @Override
    @Nonnull
    public Pair<Iterable<ContentLinkEntity>, Iterable<UnauthenticatedRemoteApplication>> getAllRemoteContentLinksAndUnauthedApps(@Nonnull String key, @Nonnull final TypeId entityTypeId)
    {
        return transformResults(executeRemoteContentLinksCollector(key, entityTypeId, new Function<ContentLinkCapability, Callable<Collection<Either<ContentLinkEntity, UnauthenticatedRemoteApplication>>>>()
        {
            @Override
            public Callable<Collection<Either<ContentLinkEntity, UnauthenticatedRemoteApplication>>> apply(final ContentLinkCapability contentLinkCapability)
            {
                return new Callable<Collection<Either<ContentLinkEntity, UnauthenticatedRemoteApplication>>>()
                {
                    public Collection<Either<ContentLinkEntity, UnauthenticatedRemoteApplication>> call()
                    {
                        final ApplicationLink applicationLink = contentLinkCapability.getEntityLink().getApplicationLink();
                        try
                        {
                            ContentLinkClient client = new ContentLinkClient(userAgentProperty);
                            return Collections2.transform(client.getContentLinks(contentLinkCapability), new Function<ContentLinkEntity, Either<ContentLinkEntity, UnauthenticatedRemoteApplication>>()
                            {
                                @Override
                                public Either<ContentLinkEntity, UnauthenticatedRemoteApplication> apply(ContentLinkEntity from)
                                {
                                    return left(from);
                                }
                            });
                        }
                        catch (CredentialsRequiredException ex)
                        {
                            return Collections.singleton(Either.<ContentLinkEntity, UnauthenticatedRemoteApplication>right(new UnauthenticatedRemoteApplication(applicationLink.getId(), applicationLink.getName(), Either.<URI, String>right(contentLinkCapability.getContentLinkUrl()), ex.getAuthorisationURI())));
                        }
                        catch (Exception ex)
                        {
                            log.error("Could not get project shortcuts for entity {}\n on app {}", contentLinkCapability.getEntityLink(),
                                    applicationLink.getName());
                            log.debug("More details", ex);
                            return Collections.emptyList();
                        }
                    }
                };
            }
        }, Suppliers.<Collection<Either<ContentLinkEntity, UnauthenticatedRemoteApplication>>>ofInstance(Collections.<Either<ContentLinkEntity, UnauthenticatedRemoteApplication>>emptyList())));
    }

    private <T> Collection<T> executeRemoteContentLinksCollector(String key, TypeId entityTypeId, Function<ContentLinkCapability, Callable<Collection<T>>> collector, Supplier<Collection<T>> emptyTsSupplier)
    {
        final EntityType entityType = typeAccessor.loadEntityType(entityTypeId);
        if (entityType != null)
        {
            final Iterable<RemoteApplicationWithCapabilities> applications = remoteApplications.capableOf(CapabilityKey.CONTENT_LINKS);
            if (!applications.iterator().hasNext())
            {
                return emptyTsSupplier.get();
            }

            final Iterable<ContentLinkCapability> contentLinkCapabilities = ContentLinkCapability.create(applications,
                    mutatingEntityLinkService.getEntityLinksForKey(key, entityType.getClass()));

            Iterable<Callable<Collection<T>>> tasks = Iterables.transform(contentLinkCapabilities, collector);
            try
            {
                return executor.invokeAllAndGet(tasks, DaemonExecutorService.DEFAULT_TIMEOUT_MS, TimeUnit.MILLISECONDS);
            }
            catch(ExecutionException ex)
            {
                log.error("Error getting project shortcuts", ex);
            }
            catch(InterruptedException ie)
            {
                // do nothing and return the empty list
            }
        }
        return emptyTsSupplier.get();
    }

    private Pair<Iterable<ContentLinkEntity>, Iterable<UnauthenticatedRemoteApplication>> transformResults(Collection<Either<ContentLinkEntity, UnauthenticatedRemoteApplication>> results)
    {
        final Iterable<ContentLinkEntity> entities = Options.flatten(Collections2.transform(results, new Function<Either<ContentLinkEntity, UnauthenticatedRemoteApplication>, Option<ContentLinkEntity>>()
        {
            @Override
            public Option<ContentLinkEntity> apply(Either<ContentLinkEntity, UnauthenticatedRemoteApplication> from)
            {
                return from.left().toOption();
            }
        }));

        final Iterable<UnauthenticatedRemoteApplication> unauthenticatedApps = Options.flatten(Collections2.transform(results, new Function<Either<ContentLinkEntity, UnauthenticatedRemoteApplication>, Option<UnauthenticatedRemoteApplication>>()
        {
            @Override
            public Option<UnauthenticatedRemoteApplication> apply(Either<ContentLinkEntity, UnauthenticatedRemoteApplication> from)
            {
                return from.right().toOption();
            }
        }));

        return pair(entities, unauthenticatedApps);

    }

    // todo copied from plugins.  Could we just make it public?
    private <T extends ConditionalDescriptor> List<T> filterFragmentsByCondition(List<T> relevantItems, Map<String, Object> context)
    {
        if (relevantItems.isEmpty())
        {
            return relevantItems;
        }

        List<T> result = new ArrayList<T>(relevantItems);
        for (Iterator<T> iterator = result.iterator(); iterator.hasNext(); )
        {
            ConditionalDescriptor descriptor = iterator.next();
            try
            {
                if (descriptor.getCondition() != null && !descriptor.getCondition().shouldDisplay(context))
                {
                    iterator.remove();
                }
            }
            catch (Exception e)
            {
                log.error("Could not evaluate condition '" + descriptor.getCondition() + "' for descriptor: " + descriptor, e);
                iterator.remove();
            }
        }

        return result;
    }

    private List<ContentLinkModuleDescriptor> filterFragmentsByTypeId(@Nonnull List<ContentLinkModuleDescriptor> descriptors, @Nonnull TypeId entityType)
    {
        List<ContentLinkModuleDescriptor> result = new ArrayList<ContentLinkModuleDescriptor>(descriptors);
        for (Iterator<ContentLinkModuleDescriptor> iterator = result.iterator(); iterator.hasNext(); )
        {
            ContentLinkModuleDescriptor projectShortcutModuleDescriptor = iterator.next();
            final Set<TypeId> entityTypes = projectShortcutModuleDescriptor.getEntityTypes();
            // if no entity types defined, show for all.
            if (entityTypes != null && !entityTypes.isEmpty() && !entityTypes.contains(entityType)) {
                iterator.remove();
            }
        }
        return result;
    }
}
