package com.atlassian.plugins.navlink.producer.contentlinks.services;

import com.atlassian.applinks.api.ApplicationLink;
import com.atlassian.applinks.api.ApplicationLinkRequest;
import com.atlassian.applinks.api.ApplicationLinkRequestFactory;
import com.atlassian.applinks.api.ApplicationLinkResponseHandler;
import com.atlassian.applinks.api.CredentialsRequiredException;
import com.atlassian.applinks.api.EntityLink;
import com.atlassian.applinks.api.EntityType;
import com.atlassian.applinks.spi.application.IdentifiableType;
import com.atlassian.plugins.navlink.consumer.http.UserAgentProperty;
import com.atlassian.plugins.navlink.producer.contentlinks.rest.ContentLinkEntity;
import com.atlassian.plugins.navlink.producer.contentlinks.rest.ContentLinksEnvelope;
import com.atlassian.sal.api.net.Request;
import com.atlassian.sal.api.net.Response;
import com.atlassian.sal.api.net.ResponseException;
import org.codehaus.jackson.map.DeserializationConfig;
import org.codehaus.jackson.map.ObjectMapper;
import org.codehaus.jackson.map.type.CollectionType;
import org.codehaus.jackson.map.type.TypeFactory;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.annotation.Nonnull;
import javax.ws.rs.core.MediaType;
import java.io.IOException;
import java.util.Collections;
import java.util.List;

import static com.google.common.base.Preconditions.checkNotNull;

public class ContentLinkClient
{
    private static final Logger log = LoggerFactory.getLogger(ContentLinkClient.class);

    private UserAgentProperty userAgentProperty;

    public ContentLinkClient(@Nonnull final UserAgentProperty userAgentProperty)
    {
        this.userAgentProperty = checkNotNull(userAgentProperty);
    }

    @Nonnull
    public List<ContentLinkEntity> getContentLinks(@Nonnull ContentLinkCapability contentLink) throws CredentialsRequiredException
    {
        final ApplicationLink applicationLink = contentLink.getEntityLink().getApplicationLink();
        final EntityLink entityLink = contentLink.getEntityLink();

        final ApplicationLinkRequestFactory authenticatedRequestFactory = applicationLink.createAuthenticatedRequestFactory();
        try
        {
            final StringBuilder url = new StringBuilder(contentLink.getContentLinkUrl());
            if(!contentLink.getContentLinkUrl().endsWith("/"))
            {
                url.append("/");
            }
            url.append(entityLink.getKey());
            
            final EntityType type = entityLink.getType();
            if (type instanceof IdentifiableType)
            {
                url.append("?entityType=").append(((IdentifiableType) type).getId());
            }
            ApplicationLinkRequest request = authenticatedRequestFactory.createRequest(Request.MethodType.GET, url.toString());
            request.addHeader("Accept", MediaType.APPLICATION_JSON);
            request.addHeader("Content-Type", MediaType.APPLICATION_JSON);
            request.addHeader("User-Agent", userAgentProperty.get());
            return request.execute(new ResponseHandler(applicationLink));
        }
        catch (ResponseException e)
        {
            log.error("Error processing response to project shortcuts request: " + e.getMessage());
            log.debug("Stacktrace: ", e);
            return Collections.emptyList();
        }
    }

    private static class ResponseHandler implements ApplicationLinkResponseHandler<List<ContentLinkEntity>>
    {
        private static final Logger log = LoggerFactory.getLogger(ResponseHandler.class);
        private final ApplicationLink applicationLink;

        public ResponseHandler(final ApplicationLink applicationLink)
        {
            this.applicationLink = applicationLink;
        }

        @Override
        public List<ContentLinkEntity> credentialsRequired(final Response response) throws ResponseException
        {
            log.debug("Project Shortcuts API is only supported on trusted apps and 2LO connections. Skipping link for: {}", applicationLink);
            return Collections.emptyList();
        }

        @Override
        public List<ContentLinkEntity> handle(final Response response) throws ResponseException
        {
            if (response.getStatusCode() != 200)
            {
                log.debug("Got non-successful response \"{}\" from project shortcuts request from: {}", response.getStatusText(), applicationLink);
                return Collections.emptyList();
            }

            ObjectMapper mapper = new ObjectMapper();
            mapper.configure(DeserializationConfig.Feature.FAIL_ON_UNKNOWN_PROPERTIES, false);
            String responseBody = response.getResponseBodyAsString();
            
            try
            {
                if(responseBody.startsWith("["))
                {
                    CollectionType expectedType = TypeFactory.defaultInstance().constructCollectionType(List.class, ContentLinkEntity.class);
                    return (List<ContentLinkEntity>) mapper.readValue(responseBody, expectedType);
                }
                else
                {
                    ContentLinksEnvelope envelope = mapper.readValue(responseBody, ContentLinksEnvelope.class);
                    return envelope.getContentLinks();
                }
            }
            catch (IOException e)
            {
                throw new ResponseException("Could not parse response: " + responseBody, e);
            }
        }
    }
}
