package com.atlassian.plugins.navlink.consumer.menu.services;

import com.atlassian.failurecache.CacheLoader;
import com.atlassian.failurecache.ExpiringValue;
import com.atlassian.fugue.Pair;
import com.atlassian.plugins.navlink.consumer.menu.client.navigation.NavigationClient;
import com.atlassian.plugins.navlink.producer.capabilities.CapabilityKey;
import com.atlassian.plugins.navlink.producer.capabilities.RemoteApplicationWithCapabilities;
import com.atlassian.plugins.navlink.producer.navigation.ApplicationNavigationLinks;
import com.atlassian.sal.api.message.LocaleResolver;
import com.google.common.collect.ImmutableSet;
import com.google.common.collect.Sets;
import com.google.common.util.concurrent.ListenableFuture;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.Collections;
import java.util.HashSet;
import java.util.Locale;
import java.util.Set;

/**
 * Populates the cache with information about remote application' navigation links.
 * @since 3.2
 */
public class NavigationLinksCacheLoader implements CacheLoader<Pair<RemoteApplicationWithCapabilities, Locale>, ApplicationNavigationLinks>
{
    private static final Logger logger = LoggerFactory.getLogger(NavigationLinksCacheLoader.class);

    private final Set<Locale> supportedLocales = new HashSet<Locale>();

    private final RemoteApplications remoteApplications;
    private final LocaleResolver localeResolver;
    private final NavigationClient navigationClient;

    public NavigationLinksCacheLoader(final RemoteApplications remoteApplications, final LocaleResolver localeResolver, final NavigationClient navigationClient)
    {
        this.remoteApplications = remoteApplications;
        this.localeResolver = localeResolver;
        this.navigationClient = navigationClient;
    }

    @Override
    public ImmutableSet<Pair<RemoteApplicationWithCapabilities, Locale>> getKeys()
    {
        final Set<RemoteApplicationWithCapabilities> knownApplications = remoteApplications.capableOf(CapabilityKey.NAVIGATION.getKey());
        return cartesianProduct(knownApplications, Sets.union(getDefaultLocales(), supportedLocales));
    }

    @Override
    public ListenableFuture<ExpiringValue<ApplicationNavigationLinks>> loadValue(final Pair<RemoteApplicationWithCapabilities, Locale> key)
    {
        return navigationClient.getNavigationLinks(key.left(), key.right());
    }

    /**
     * Notify the cache loader about a missing locale in the cache. The cache loader itself is responsible to decide if
     * the locale will be added to the cache or not (e.g. due to size constraints).
     * @param locale the locale currently that is missing in the cache
     */
    public void cacheMissFor(final Locale locale)
    {
        if (!supportedLocales.contains(locale))
        {
            logger.debug("Adding locale '{}' to the list of supported languages when fetching navigation links.", locale);
            supportedLocales.add(locale);
        }
    }

    /**
     * @return the application's default locale and additionally #{@link Locale#UK} if the default is not
     * #{@link Locale#ENGLISH} (as fall-back).
     */
    private Set<Locale> getDefaultLocales()
    {
        final Locale locale = localeResolver.getLocale();
        return locale.getLanguage().equals(Locale.ENGLISH.getLanguage()) ? Collections.singleton(locale) : Sets.newHashSet(locale, Locale.UK);
    }

    private <L, R> ImmutableSet<Pair<L, R>> cartesianProduct(final Iterable<L> leftValues, final Iterable<R> rightValue)
    {
        final ImmutableSet.Builder<Pair<L, R>> builder = ImmutableSet.builder();
        for (final L left : leftValues)
        {
            for (final R right : rightValue)
            {
                builder.add(Pair.pair(left, right));
            }
        }
        return builder.build();
    }
}
