package com.atlassian.plugins.navlink.consumer.menu.services;

import com.atlassian.applinks.api.ApplicationLink;
import com.google.common.base.Predicate;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.annotation.concurrent.Immutable;

/**
 * Returns <code>true</code> if the given {@link ApplicationLink} has a magic flag which indicates that it was created
 * by the remote plugin and it doesn't want that we query the application link for capabilities.
 */
@Immutable
public enum IgnoreRemotePluginApplicationLinkPredicate implements Predicate<ApplicationLink>
{
    INSTANCE;

    /**
     * A remote plugin sets this magic property to <code>false</code> on an application link to avoid streams to follow
     * the link.
     *
     * @see <a href="https://ecosystem.atlassian.net/browse/STRM-1446">STRM-1446</a>
     */
    private static final String MAGIC_PROPERTY = "IS_ACTIVITY_ITEM_PROVIDER";

    @Override
    public boolean apply(@Nullable final ApplicationLink input)
    {
        return input != null && hasMagicPropertySetToFalse(input);
    }

    private boolean hasMagicPropertySetToFalse(@Nonnull final ApplicationLink applicationLink)
    {
        final String propertyValue = getStringProperty(applicationLink, MAGIC_PROPERTY);
        return propertyValue != null && propertyValue.equalsIgnoreCase(Boolean.FALSE.toString());
    }

    @Nullable
    private String getStringProperty(@Nonnull final ApplicationLink applicationLink, @Nonnull final String propertyKey)
    {
        final Object value = applicationLink.getProperty(propertyKey);
        return value != null && value instanceof String ? String.class.cast(value) : null;
    }
}