package com.atlassian.plugins.navlink.consumer.menu.services;

import com.atlassian.failurecache.Cache;
import com.atlassian.failurecache.CacheFactory;
import com.atlassian.failurecache.Cacheable;
import com.atlassian.failurecache.Refreshable;
import com.atlassian.plugins.navlink.producer.navigation.ApplicationNavigationLinks;
import com.atlassian.plugins.navlink.producer.navigation.NavigationLink;
import com.atlassian.plugins.navlink.util.executor.DaemonExecutorService;
import com.google.common.base.Function;
import com.google.common.base.Predicate;
import com.google.common.base.Predicates;
import com.google.common.collect.Collections2;
import com.google.common.collect.ImmutableSet;
import com.google.common.util.concurrent.Futures;
import com.google.common.util.concurrent.ListenableFuture;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.InitializingBean;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import java.util.Collection;
import java.util.Collections;
import java.util.Locale;
import java.util.Set;
import java.util.concurrent.TimeUnit;

import static com.google.common.collect.Iterables.concat;
import static com.google.common.collect.Iterables.filter;
import static com.google.common.collect.Iterables.transform;

/**
 * Serves information about remote application's navigation links directly from a cache. Updates
 * @since 3.2
 */
public class CachingRemoteNavigationLinkServiceImpl
        implements Cacheable, InitializingBean, Runnable, Refreshable, RemoteNavigationLinkService
{
    private static final long INITIAL_DELAY_IN_SECONDS = Long.getLong("navlink.navigationlinkscache.initialdelay", 35);
    private static final long DELAY_IN_SECONDS = Long.getLong("navlink.navigationlinkscache.delay", 10);
    private static final Logger logger = LoggerFactory.getLogger(CachingRemoteNavigationLinkServiceImpl.class);

    private final NavigationLinksCacheLoader navigationLinksCacheLoader;
    private final Cache<ApplicationNavigationLinks> cache;
    private final DaemonExecutorService executorService;

    @SuppressWarnings("UnusedDeclaration")
    public CachingRemoteNavigationLinkServiceImpl(final NavigationLinksCacheLoader navigationLinksCacheLoader, final DaemonExecutorService executorService, final CacheFactory cacheFactory)
    {
        this.navigationLinksCacheLoader = navigationLinksCacheLoader;
        this.cache = cacheFactory.createExpirationDateBasedCache(navigationLinksCacheLoader);
        this.executorService = executorService;
    }

    @Override
    @Nonnull
    public Set<NavigationLink> all(@Nonnull final Locale locale)
    {
        return matching(locale, Predicates.<NavigationLink>alwaysTrue());
    }

    @Override
    @Nonnull
    public Set<NavigationLink> matching(@Nonnull final Locale locale, @Nullable final Predicate<NavigationLink> criteria)
    {
        final Iterable<ApplicationNavigationLinks> applicationNavigationLinksForLocale = filterCacheByLocale(locale);
        final Iterable<ImmutableSet<NavigationLink>> navigationLinkSetsForLocale = transform(applicationNavigationLinksForLocale, extractNavigationLinkSets());
        final Iterable<NavigationLink> navigationLinksForLocale = concat(navigationLinkSetsForLocale);
        final Iterable<NavigationLink> filteredNavigationLinksForLocale = filter(navigationLinksForLocale, criteria);
        return ImmutableSet.copyOf(filteredNavigationLinksForLocale);
    }

    @Override
    public void run()
    {
        refreshCache();
    }

    @Override
    public int getCachePriority()
    {
        return 700;
    }

    @Override
    public void clearCache()
    {
        cache.clear();
    }

    @Override
    public ListenableFuture<?> refreshCache()
    {
        try
        {
            return cache.refresh();
        }
        catch (Exception e)
        {
            logger.debug("Failed to refresh remote menu items cache", e);
            return Futures.immediateFailedFuture(e);
        }
    }

    @Override
    public void afterPropertiesSet() throws Exception
    {
        executorService.scheduleWithFixedDelay(this, INITIAL_DELAY_IN_SECONDS, DELAY_IN_SECONDS, TimeUnit.SECONDS);
    }

    @SuppressWarnings("unchecked")
    private Iterable<ApplicationNavigationLinks> filterCacheByLocale(final Locale mostSpecificLocale)
    {
        final ImmutableSet<ApplicationNavigationLinks> cacheValues = ImmutableSet.copyOf(cache.getValues());
        final Collection<ApplicationNavigationLinks> expectedCacheHit = Collections2.filter(cacheValues, filterByLocale(mostSpecificLocale));
        if (!expectedCacheHit.isEmpty())
        {
            return expectedCacheHit;
        }
        else
        {
            navigationLinksCacheLoader.cacheMissFor(mostSpecificLocale);
            final Locale sameLanguage = new Locale(mostSpecificLocale.getLanguage());
            return filterWithFallBack(cacheValues,
                    filterByLanguage(sameLanguage),
                    filterByLanguage(Locale.ENGLISH) // assuming all our products ship at least with English
            );
        }
    }

    private Iterable<ApplicationNavigationLinks> filterWithFallBack(final ImmutableSet<ApplicationNavigationLinks> cacheValues, final Predicate<ApplicationNavigationLinks>... filterFunctions)
    {
        for (final Predicate<ApplicationNavigationLinks> filterFunction : filterFunctions)
        {
            final Collection<ApplicationNavigationLinks> applicationNavigationLinksForLocale = Collections2.filter(cacheValues, filterFunction);
            if (!applicationNavigationLinksForLocale.isEmpty())
            {
                return applicationNavigationLinksForLocale;
            }

        }
        return Collections.emptySet();
    }

    private Predicate<ApplicationNavigationLinks> filterByLocale(final Locale locale)
    {
        return new Predicate<ApplicationNavigationLinks>()
        {
            @Override
            public boolean apply(@Nullable final ApplicationNavigationLinks input)
            {
                return input != null && input.getLocale().equals(locale);
            }
        };
    }

    private Predicate<ApplicationNavigationLinks> filterByLanguage(final Locale locale)
    {
        final String language = locale.getLanguage();
        return new Predicate<ApplicationNavigationLinks>()
        {

            @Override
            public boolean apply(@Nullable final ApplicationNavigationLinks input)
            {
                return input != null && input.getLocale().getLanguage().equals(language);
            }
        };
    }


    private Function<ApplicationNavigationLinks, ImmutableSet<NavigationLink>> extractNavigationLinkSets()
    {
        return new Function<ApplicationNavigationLinks, ImmutableSet<NavigationLink>>()
        {
            @Override
            public ImmutableSet<NavigationLink> apply(@Nullable final ApplicationNavigationLinks from)
            {
                return from != null ? from.getNavigationLinks() : ImmutableSet.<NavigationLink>of();
            }
        };
    }
}
