package com.atlassian.plugins.navlink.consumer.menu.client.navigation;

import com.atlassian.plugins.navlink.producer.capabilities.RemoteApplicationWithCapabilities;
import com.atlassian.plugins.navlink.producer.navigation.ApplicationNavigationLinks;
import com.atlassian.plugins.navlink.producer.navigation.NavigationLink;
import com.atlassian.plugins.navlink.producer.navigation.NavigationLinkBuilder;
import com.atlassian.plugins.navlink.producer.navigation.links.LinkSource;
import com.google.common.base.Strings;
import org.apache.http.HttpResponse;
import org.apache.http.client.ResponseHandler;
import org.apache.http.impl.client.BasicResponseHandler;
import org.codehaus.jackson.JsonNode;
import org.codehaus.jackson.map.ObjectMapper;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.IOException;
import java.util.Collections;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Locale;
import java.util.Set;
import javax.annotation.Nullable;

public class NavigationLinkResponseHandler implements ResponseHandler<ApplicationNavigationLinks>
{
    private static final Logger logger = LoggerFactory.getLogger(NavigationLinkResponseHandler.class);

    private static final String WEIGHT_FIELD = "weight";
    private static final String LINKS_FIELD = "links";
    private static final String BASE_FIELD = "base";

    private final ResponseHandler<String> basicResponseHandler = new BasicResponseHandler();
    private final String applicationType;
    private final String applicationId;
    private final Locale locale;

    public NavigationLinkResponseHandler(final RemoteApplicationWithCapabilities application, final Locale locale)
    {
        applicationType = application.getType();
        applicationId = application.getApplicationLinkId();
        this.locale = locale;
    }

    @Override
    public ApplicationNavigationLinks handleResponse(final HttpResponse response) throws IOException
    {
        return decodeResponse(basicResponseHandler.handleResponse(response));
    }

    private ApplicationNavigationLinks decodeResponse(@Nullable final String responseBody) throws IOException
    {
        if (responseBody == null)
        {
            return new ApplicationNavigationLinks(locale, Collections.<NavigationLink>emptySet());
        }

        return parseBody(responseBody);
    }

    protected ApplicationNavigationLinks parseBody(final String responseBody) throws IOException
    {
        final ObjectMapper mapper = new ObjectMapper();
        final JsonNode jsonNodes = mapper.readValue(responseBody, JsonNode.class);
        final String baseUrl = parseBaseUrl(jsonNodes);
        final Set<NavigationLink> result = parseLinks(jsonNodes, baseUrl);
        return new ApplicationNavigationLinks(locale, result);
    }

    private String parseBaseUrl(final JsonNode jsonNodes)
    {
        final String baseUrl = jsonNodes.path(LINKS_FIELD).path(BASE_FIELD).getTextValue();
        return Strings.emptyToNull(baseUrl);
    }

    private Set<NavigationLink> parseLinks(final JsonNode jsonNodes, final String baseUrl)
    {
        final Set<NavigationLink> result = new HashSet<NavigationLink>();
        final Iterator<String> fieldNames = jsonNodes.getFieldNames();
        while (fieldNames.hasNext())
        {
            final String fieldName = fieldNames.next();
            if (!fieldName.equals(LINKS_FIELD) && jsonNodes.path(fieldName).isArray())
            {
                Iterator<JsonNode> elements = jsonNodes.path(fieldName).getElements();
                while (elements.hasNext())
                {
                    final NavigationLink navigationLink = createNavigationLink(fieldName, elements.next(), baseUrl);
                    if (navigationLink != null)
                    {
                        result.add(navigationLink);
                    }
                }
            }
        }
        return result;
    }

    @Nullable
    private NavigationLink createNavigationLink(final String menuKey, final JsonNode navigationLinkNode, final String baseUrl)
    {
        final String href = navigationLinkNode.path("href").getTextValue();
        final String label = navigationLinkNode.path("label").getTextValue();
        final String icon = navigationLinkNode.path("icon").getTextValue();
        final String tooltip = navigationLinkNode.path("tooltip").getTextValue();
        if (!Strings.isNullOrEmpty(href) && !Strings.isNullOrEmpty(label))
        {
            return new NavigationLinkBuilder()
                    .key(menuKey)
                    .href(href)
                    .baseUrl(baseUrl)
                    .iconUrl(icon)
                    .label(label)
                    .tooltip(tooltip)
                    .applicationType(applicationType)
                    .source(LinkSource.remote(applicationId))
                    .weight(parseWeight(navigationLinkNode))
                    .build();
        }
        else
        {
            return null;
        }
    }

    private int parseWeight(JsonNode jsonNodes)
    {
        final JsonNode weightNode = jsonNodes.get(WEIGHT_FIELD);
        if (weightNode != null)
        {
            if (weightNode.isNumber())
            {
                return weightNode.asInt();
            }
            else
            {
                logger.warn("Encountered non-numeric weight property in parsed JSON response");
                logger.debug("Non-numeric weight property in response " + jsonNodes.toString());
            }
        }
        return Integer.MAX_VALUE;
    }

}
