package com.atlassian.plugins.navlink.consumer.http;

import com.atlassian.failurecache.ExpiringValue;
import com.atlassian.plugins.navlink.consumer.http.caching.HttpCacheExpiryService;
import org.apache.http.HttpResponse;
import org.apache.http.client.ResponseHandler;

import java.io.IOException;

import static com.google.common.base.Preconditions.checkNotNull;

/**
 * Extract any cache information from the http response and encapsulates them into an {@link ExpiringValue}. Doesn't do
 * any error handling.
 * @param <T> the type being wrapped into an {@link ExpiringValue}
 * @since 3.2
 */
public class ExpiringValueResponseHandler<T> implements ResponseHandler<ExpiringValue<T>>
{
    private final ResponseHandler<T> delegatee;
    private final HttpCacheExpiryService httpCacheExpiryService;

    public ExpiringValueResponseHandler(final ResponseHandler<T> delegatee, final HttpCacheExpiryService httpCacheExpiryService)
    {
        this.delegatee = checkNotNull(delegatee);
        this.httpCacheExpiryService = checkNotNull(httpCacheExpiryService);
    }

    @Override
    public ExpiringValue<T> handleResponse(final HttpResponse response) throws IOException
    {
        final T content = delegatee.handleResponse(response);
        return httpCacheExpiryService.createExpiringValueFrom(response, content);
    }
}
