package com.atlassian.plugins.navlink.consumer;

import com.atlassian.applinks.api.ApplicationId;
import com.atlassian.applinks.api.ApplicationLink;
import com.atlassian.applinks.api.ApplicationLinkRequestFactory;
import com.atlassian.applinks.api.ApplicationLinkService;
import com.atlassian.applinks.api.ApplicationType;
import com.atlassian.applinks.api.auth.AuthenticationProvider;
import com.atlassian.sal.api.transaction.TransactionCallback;
import com.atlassian.sal.api.transaction.TransactionTemplate;
import com.google.common.base.Function;
import com.google.common.base.Predicate;
import com.google.common.base.Supplier;
import com.google.common.cache.Cache;
import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheLoader;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableMap;

import javax.annotation.Nullable;
import java.net.URI;
import java.util.List;
import java.util.Map;
import java.util.concurrent.TimeUnit;

import static com.google.common.base.Preconditions.checkNotNull;
import static com.google.common.collect.Iterables.filter;
import static com.google.common.collect.Iterables.getOnlyElement;
import static com.google.common.collect.Iterables.transform;

public class CachingApplicationLinkServiceImpl implements CachingApplicationLinkService
{
    private final ApplicationLinkService applicationLinkService;
    private final TransactionTemplate transactionTemplate;
    // ROTP-975 expire entries after some time to pick up applinks created by applinks 2 code
    private final Cache<Object, Iterable<ApplicationLink>> links = CacheBuilder.newBuilder().expireAfterWrite(5, TimeUnit.MINUTES).build(CacheLoader.from(new Supplier<Iterable<ApplicationLink>>()
    {
        @Override
        public Iterable<ApplicationLink> get()
        {
            return transactionTemplate.execute(new TransactionCallback<Iterable<ApplicationLink>>() {
                @Override
                public Iterable<ApplicationLink> doInTransaction()
                {
                    return ImmutableList.copyOf(transform(applicationLinkService.getApplicationLinks(),
                        new Function<ApplicationLink, ApplicationLink>()
                        {
                            @Override
                            public ApplicationLink apply(@Nullable ApplicationLink from)
                            {
                                return new ImmutableApplicationLink(from);
                            }
                        }));
                }
            });
        }
    }));

    public CachingApplicationLinkServiceImpl(ApplicationLinkService applicationLinkService,
                                             TransactionTemplate transactionTemplate)
    {
        this.applicationLinkService = applicationLinkService;
        this.transactionTemplate = transactionTemplate;
    }

    @Override
    public Iterable<ApplicationLink> getApplicationLinks()
    {
        return links.getUnchecked(this);
    }

    @Override
    public ApplicationLink getApplicationLink(final ApplicationId applicationId)
    {
        checkNotNull(applicationId);
        return getOnlyElement(filter(getApplicationLinks(), new Predicate<ApplicationLink>()
        {
            @Override
            public boolean apply(ApplicationLink input)
            {
                return applicationId.equals(input.getId());
            }
        }), null);
    }

    @Override
    public void clear()
    {
        links.invalidateAll();
    }

    private static class ImmutableApplicationLink implements ApplicationLink
    {
        private static final List<String> KNOWN_PROPERTY_KEYS = ImmutableList.of("IS_ACTIVITY_ITEM_PROVIDER");
        private final ApplicationId applicationId;
        private final ApplicationType applicationType;
        private final String name;
        private final URI displayUrl;
        private final URI rpcUrl;
        private final Map<String, Object> properties;

        public ImmutableApplicationLink(ApplicationLink that)
        {
            this.applicationId = that.getId();
            this.applicationType = that.getType();
            this.name = that.getName();
            this.displayUrl = that.getDisplayUrl();
            this.rpcUrl = that.getRpcUrl();
            ImmutableMap.Builder<String, Object> builder = ImmutableMap.builder();
            for (String knownPropertyKey : KNOWN_PROPERTY_KEYS)
            {
                final Object property = that.getProperty(knownPropertyKey);
                if (property != null)
                {
                    builder.put(knownPropertyKey, property);
                }
            }
            this.properties = builder.build();
        }

        @Override
        public ApplicationId getId()
        {
            return applicationId;
        }

        @Override
        public ApplicationType getType()
        {
            return applicationType;
        }

        @Override
        public String getName()
        {
            return name;
        }

        @Override
        public URI getDisplayUrl()
        {
            return displayUrl;
        }

        @Override
        public URI getRpcUrl()
        {
            return rpcUrl;
        }

        @Override
        public boolean isPrimary()
        {
            throw new UnsupportedOperationException("isPrimary not cached");
        }

        @Override
        public boolean isSystem()
        {
            throw new UnsupportedOperationException("isSystem not cached");
        }

        @Override
        public ApplicationLinkRequestFactory createAuthenticatedRequestFactory()
        {
            throw new UnsupportedOperationException("createAuthenticatedRequestFactory not available");
        }

        @Override
        public ApplicationLinkRequestFactory createAuthenticatedRequestFactory(
            Class<? extends AuthenticationProvider> providerClass)
        {
            throw new UnsupportedOperationException("createAuthenticatedRequestFactory not available");
        }

        @Override
        public ApplicationLinkRequestFactory createImpersonatingAuthenticatedRequestFactory()
        {
            throw new UnsupportedOperationException("createImpersonatingAuthenticatedRequestFactory not available");
        }

        @Override
        public ApplicationLinkRequestFactory createNonImpersonatingAuthenticatedRequestFactory()
        {
            throw new UnsupportedOperationException("createNonImpersonatingAuthenticatedRequestFactory not available");
        }

        @Override
        public Object getProperty(String key)
        {
            if (!KNOWN_PROPERTY_KEYS.contains(key))
                throw new UnsupportedOperationException("Property " + key + " not cached");
            return properties.get(key);
        }

        @Override
        public Object putProperty(String key, Object value)
        {
            throw new UnsupportedOperationException("putProperty not allowed on immutable applink");
        }

        @Override
        public Object removeProperty(String key)
        {
            throw new UnsupportedOperationException("removeProperty not allowed on immutable applink");
        }

        @Override
        public String toString()
        {
            return String.format("%s (%s) %s %s", name, applicationId, rpcUrl, applicationType);
        }
    }
}
