package com.atlassian.plugins.custom_apps.api;

import com.atlassian.plugins.navlink.consumer.menu.services.NavigationLinkComparator;
import com.atlassian.plugins.navlink.producer.navigation.NavigationLink;
import com.atlassian.plugins.navlink.producer.navigation.NavigationLinkBuilder;
import com.google.common.base.Objects;
import com.google.common.collect.Lists;

import javax.annotation.Nonnull;
import javax.annotation.concurrent.Immutable;
import java.util.List;

import static com.google.common.base.Preconditions.checkNotNull;

/**
 * An immutable custom app.
 *
 * @since 1.0
 */
@Immutable
public final class CustomApp
{
    private final String id;
    private final NavigationLink navigationLink;
    private final String sourceApplicationUrl;
    private final String sourceApplicationName;
    private final List<String> allowedGroups;
    private final boolean editable;
    private final boolean hide;

    /**
     *
     * @param id unique ID of the CustomApp
     * @param link a {NavigationLink}
     * @param sourceApplicationUrl the url of the source application, this should be null for local links and contain the remote application url for remote links.
     * @param sourceApplicationName the display name of the source application.
     * @param hide flag indicating if the custom app is hidden form the app switcher for all users.
     * @param allowedGroups if the custom app is not hidden from all users a set of allow groups will restrict its visibility to members of those groups only.
     * @param editable flag indicating if the custom app entry is editable or not. Only locally created Custom Apps should be editable. Not remote entries or the local application link.
     */
    public CustomApp(@Nonnull final String id, @Nonnull final NavigationLink link, final String sourceApplicationUrl, final String sourceApplicationName, final boolean hide, final List<String> allowedGroups, final boolean editable)
    {
        this.id = checkNotNull(id);
        this.navigationLink = checkNotNull(link);
        this.hide = hide;

        if(allowedGroups != null)
        {
            this.allowedGroups = Lists.newArrayList(allowedGroups);
        }
        else
        {
            this.allowedGroups = Lists.newArrayList();
        }
        this.sourceApplicationUrl = sourceApplicationUrl;
        this.sourceApplicationName = sourceApplicationName;
        this.editable = editable;
    }

    /**
     *
     * @param id unique ID of the CustomApp
     * @param displayName the display name of the link
     * @param url the URL of the link
     * @param sourceApplicationUrl the url of the source application, this should be null for local links and contain the remote application url for remote links.
     * @param sourceApplicationName the display name of the source application.
     * @param sourceApplicationType the type of the source application e.g. JIRA
     * @param hide flag indicating if the custom app is hidden form the app switcher for all users.
     * @param allowedGroups if the custom app is not hidden from all users a set of allow groups will restrict its visibility to members of those groups only.
     * @param editable flag indicating if the custom app entry is editable or not. Only locally created Custom Apps should be editable. Not remote entries or the local application link.
     * @param self if the link represents the current application
     */
    public CustomApp(@Nonnull final String id, @Nonnull final String displayName, @Nonnull final String url, String sourceApplicationUrl, String sourceApplicationName, String sourceApplicationType, boolean hide, List<String> allowedGroups, final boolean editable, boolean self)
    {
        this(
                id,
                new NavigationLinkBuilder()
                        .key(NavigationLink.CUSTOM_APPS_KEY)
                        .href(checkNotNull(url))
                        .label(checkNotNull(displayName))
                        .applicationType(sourceApplicationType)
                        .self(self)
                        .weight(NavigationLinkComparator.Weights.MAX.value())
                        .build(),
                sourceApplicationUrl,
                sourceApplicationName,
                hide,
                allowedGroups,
                editable
        );
    }

    public CustomApp(@Nonnull final String id, @Nonnull final String displayName, @Nonnull final String url, String sourceApplicationUrl, String sourceApplicationName, String sourceApplicationType, boolean hide, List<String> allowedGroups, final boolean editable)
    {
        this(id, displayName, url, sourceApplicationUrl, sourceApplicationName, sourceApplicationType, hide, allowedGroups, editable, false);
    }

    @Nonnull
    public String getId()
    {
        return id;
    }

    @Nonnull
    public String getDisplayName()
    {
        return navigationLink.getLabel();
    }

    @Nonnull
    public String getUrl()
    {
        return navigationLink.getHref();
    }

    public String getSourceApplicationUrl()
    {
        return sourceApplicationUrl;
    }

    public String getSourceApplicationName()
    {
        return sourceApplicationName;
    }

    public String getSourceApplicationType()
    {
        return navigationLink.getApplicationType();
    }

    /**
     * Gets a flag indicating if this instance is hidden from all users.
     * If true visibility may still be determined by allowedGroups.
     *
     * @return true if the link should hidden, false otherwise
     */
    public boolean getHide()
    {
        return hide;
    }

    /**
     * Gets the groups a user must belong to to view this instance, if it is not hidden from all users. see getHide()
     * An empty list signifies unrestricted.
     * @return a list of group names
     */
    public List<String> getAllowedGroups()
    {
        return allowedGroups;
    }

    public boolean getEditable()
    {
        return editable;
    }

    public boolean isSelf()
    {
        return navigationLink.isSelf();
    }

    public NavigationLink getNavigationLink()
    {
        return navigationLink;
    }

    @Override
    public int hashCode()
    {
        return Objects.hashCode(id, getDisplayName(), getUrl());
    }

    @Override
    public boolean equals(Object obj)
    {
        if (obj == this)
        {
            return true;
        }

        if (!(obj instanceof CustomApp))
        {
            return false;
        }

        final CustomApp that = (CustomApp) obj;
        return Objects.equal(this.getId(), that.getId()) && Objects.equal(this.getDisplayName(), that.getDisplayName()) && Objects.equal(this.getUrl(), that.getUrl());
    }

    @Override
    public String toString()
    {
        return "CustomApp {id='" + id + "', displayName='" + getDisplayName() + "'" + ", url='" + getUrl() + "', sourceApplicationUrl='" + sourceApplicationUrl + "', sourceApplicationName='" + sourceApplicationName + "'" + ", sourceApplicationType='"+ getSourceApplicationType() + "'" + ", hide='" + hide + ", editable='" + editable + "'}";
    }
}
