//based on mbostock's http://bl.ocks.org/mbostock/raw/7607999/
var dependencies = (function(d3, AJS, $) {
    var module = {},
        bundleToPlugins,
        currentlySelected = undefined,
        tabHandlers = {
            registeredServices: updateServices,
            servicesInUse: updateServices,
            providedWires: updateWiring,
            requiredWires: updateWiring,
            declaredRequirements: updateDeclaredRequirements,
            declaredCapabilities: updateDeclaredCapabilities
        },
        connFilters = [
            {name: "All", filter: ['dynamic','optional','mandatory','service']},
            {name: "Mandatory", filter: ['mandatory']},
            {name: "Optional", filter: ['optional']},
            {name: "Dynamic", filter: ['dynamic']},
            {name: "Wires", filter: ['dynamic','optional','mandatory']},
            {name: "Services", filter: ['service']}
        ],
        currentConnFilter = connFilters[0], // All by default
        currentPackageFilter = "";

    module.updateFilters = function (){
        var dropdown = d3.select("#display-selection");
        dropdown.selectAll("ul").remove();
        dropdown.append("ul")
            .attr("class", "aui-list-truncate")
            .selectAll("li")
            .data(connFilters)
            .enter()
            .append("li")
            .append("a")
            .attr("href", "#")
            .on("click", function(d){
                currentConnFilter = d;
                module.display();
            })
            .text(function(d){return d.name});

        $("#package-filter" ).submit(function( event ) {
            event.preventDefault();
            currentPackageFilter = $("input#quicksearchid").val();
            module.display();
            return true;
        });

        $(document).on("tabSelect", updateTab);
    };

    module.display = function () {
        var diameter = 1000,
            radius = diameter / 2,
            innerRadius = radius - 300;

        currentlySelected = undefined;

        var pluginsResourcePath = "/rest/depview/1.0/bundles";

        var cluster = d3.layout.cluster()
            .size([360, innerRadius])
            .sort(null)
            .value(function (d) { return d.size; });

        var bundle = d3.layout.bundle();

        var line = d3.svg.line.radial()
            .interpolate("bundle")
            .tension(.85)
            .radius(function (d) { return d.y; })
            .angle(function (d) { return d.x / 180 * Math.PI; });

        var info = d3.select("#current-filter");
        info.selectAll("*").remove();
        info
            .append("h3")
            .text(`Showing "${currentConnFilter.name}" for package filter "${currentPackageFilter}"`);

        var panel = d3.select("#diagram");

        panel.selectAll("svg").remove();

        var svg = panel.append("svg")
            .attr("width", diameter)
            .attr("height", diameter)
            .append("g")
            .attr("transform", "translate(" + radius + "," + radius + ")");

        var link = svg.append("g").selectAll(".link"),
            node = svg.append("g").selectAll(".node");

        var bundlesUrl=AJS.contextPath()+pluginsResourcePath+"?q="+currentPackageFilter;
        //bundlesUrl="http://localhost:8000"+pluginsResourcePath+".json";

        d3.json(bundlesUrl, function (error, bundles) {
            if (error) {
                errorMessage("bundle list", error);
                return;
            }
            //bundles.links.plugins = "http://localhost:8000"+pluginsResourcePath+"/plugins.json";
            d3.json(bundles.links.plugins, function(error, plugins) {
                if (error) {
                    errorMessage("bundle to plugin mapping", error);
                    return;
                }
                bundleToPlugins = plugins;

                var nodes = cluster.nodes(packageHierarchy(bundles.bundles)),
                    links = pluginDeps(nodes, currentConnFilter.filter);

                link = link
                    .data(bundle(links.deps))
                    .enter().append("path")
                    .each(function (d) {
                        d.source = d[0];
                        d.target = d[d.length - 1]
                    })
                    .attr("class", function (d) {
                        var c = "link ";
                        links.resolutionTypes[d.source.bundle.bundleId + "->" + d.target.bundle.bundleId].forEach(function (t) {
                            c = c + " link--type--" + t;
                        });
                        return c;
                    })
                    .attr("d", line);

                node = node
                    .data(nodes.filter(function (n) { return !n.children; }))
                    .enter();

                nodeText(node, pluginLabel, "0")
                    .attr("class", "node--label");


                node = nodeText(node, pluginText, "2em")
                    .attr("class", "node")
                    .on("mouseover", mouseovered)
                    .on("mouseout", mouseouted)
                    .on("click", onclick);
            });
        });

        function nodeText(node, text, offsetX) {
            return node
                .append("text")
                .attr("dy", ".31em")
                .attr("dx", function(d) {return d.x<180?offsetX: "-"+offsetX})
                .attr("transform", function (d) { return "rotate(" + (d.x - 90) + ")translate(" + (d.y + 8) + ",0)" + (d.x < 180 ? "" : "rotate(180)"); })
                .style("text-anchor", function (d) { return d.x < 180 ? "start" : "end"; })
                .text(text);
        }

        function addLabelModifier(d,label,state,m) {
            if (d.bundle.state == state) {
                return d.x < 180 ? label+m : m+label;
            }
            return label;
        }

        function pluginLabel(d) {
            var label;

            switch (bundleToPlugins[d.bundle.bundleId].pluginClass) {
                case "OsgiBundlePlugin":
                    label = "BP";
                    break;
                case "OsgiPlugin":
                    label = "P";
                    break;
                case "":
                    label = "B";
                    break;
                default:
                    label = "?";
            }

            label = addLabelModifier(d,label,"INSTALLED","-");
            label = addLabelModifier(d,label,"RESOLVED","+");

            return label;
        }

        function pluginText(d) {
            return d.pluginKey
                .replace(/com\./g, "c.")
                .replace(/org\./g, "o.")
                .replace(/\.apache\./g, ".a.")
                .replace(/\.atlassian\./g, ".a.")
                .replace(/\.springsource\./g, ".s.")
                .replace(/servicemix\.bundles/g, "s.b")
                .replace(/\.eclipse\./g, ".e.")
                .replace(/confluence/g, "c")
                .replace(/plugins/g, "p")
                .replace(/\.jira/g, ".j");
        }


        function highlight (d) {
            node
                .each(function (n) { n.target = n.source = false; });

            link
                .classed("link--target", function (l) { if (l.target === d) return l.source.source = true; })
                .classed("link--source", function (l) { if (l.source === d) return l.target.target = true; })
                .filter(function (l) { return l.target === d || l.source === d; })
                .each(function () { this.parentNode.appendChild(this); });

            node
                .classed("node--target", function (n) { return n.target; })
                .classed("node--source", function (n) { return n.source; })
                .classed("node--selected", function (n) { return n === d });
        }

        function original () {
            link
                .classed("link--target link--source", false);

            node
                .classed("node--target node--source node--selected", false);
        }

        function mouseovered(d) {if (!currentlySelected) highlight(d)}
        function mouseouted(d) {if (!currentlySelected) original(d)}

        function onclick(d) {
            var prev_disHover = currentlySelected;
            currentlySelected=d;
            original();
            highlight(d);
            if (prev_disHover===d) {
                currentlySelected = undefined;
            }
        }
    };

    function removePluginInfo(section) {
        d3.select("#bundle-"+section).selectAll("*").remove();
    }

    function updateSection (section) {
        removePluginInfo(section);

        const sectionDiv = d3.select("#bundle-" + section);

        if (!currentlySelected) {
            sectionDiv.append('span')
                .text('You need to first pick a bundle from the diagram under the "Diagram" tab')
        }

        d3.json(currentlySelected.bundle.links[section], function (error, result) {
            if (error) {
                errorMessage(section, error);
                return;
            }

            sectionDiv.append("h2")
                .text(currentlySelected.pluginKey);

            tabHandlers[section](sectionDiv, result, section);
        });
    }

    function updateServices(servicesDiv, services, section) {
        if (section == "registeredServices") {
            printServicesByPlugin(servicesDiv, services);
        } else {
            servicesDiv.append("ul").selectAll("li")
                .data(groupBy(services, function (s) {return bundleToPlugins[s.bundleId].pluginKey}))
                .enter().append("li")
                .each(function(d){
                    var target = d3.select(this);

                    target
                        .append("p").append("h3")
                        .text(function (d) {return d.group});

                    return printServicesByPlugin(target, d.groupped)
                });
        }

        function printServicesByPlugin(target, serviceByPlugin) {
            target.append("ul").selectAll("li")
                .data(serviceByPlugin)
                .enter().append("li")
                .each(function(service){
                    service_li = d3.select(this);

                    service_li.append("ul").selectAll("li")
                        .data(service.properties.objectClass.sort())
                        .enter().append("li")
                        .append("strong")
                        .text(function(d){return d});

                    var table = service_li.append("table").attr("class", "aui");

                    var thead_tr = table.append("thead").append("tr");
                    thead_tr.append("th").attr("id", "property_name").text("Property name");
                    thead_tr.append("th").attr("id", "property_value").text("value");

                    table.append("tbody").selectAll("tr")
                        .data(Object.keys(service.properties).filter(function(p){return p!="objectClass"}).sort())
                        .enter().append("tr")
                        .each(function(p){
                            var tr = d3.select(this);

                            tr.append("td").append("i").text(p);
                            tr.append("td").text(service.properties[p]);
                        });

                    if (section == "registeredServices" && service.usingBundles.length) {
                        service_li.append("p").append("strong").text("Used By:");
                        service_li.append("ul").selectAll("li")
                            .data(service.usingBundles.map(function(id){return bundleToPlugins[id].pluginKey}).sort())
                            .enter().append("li")
                            .text(function(d){return d});
                    }
                    service_li.append("hr");
                });

        }
    }

    function updateWiringUni(wiringDiv, wiring, groupByFunc, textFunc) {
        wiringDiv.append("ul").selectAll("li")
            .data(wiresByTypeToTypes(wiring))
            .enter().append("li")
            .each(function(d) {
                var type_li = d3.select(this);
                type_li
                    .append("p")
                    .text(function (d) {return d.type});

                type_li
                    .append("ul").selectAll("li")
                    .data(groupBy(d.wires, groupByFunc))
                    .enter().append("li")
                    .each(function (wired_plugin) {
                        target = d3.select(this);

                        target
                            .append("p")
                            .text(function (d) {return d.group});

                        target
                            .append("ul").selectAll("li")
                            .data(wired_plugin.groupped)
                            .enter()
                            .append("li").append("div")
                            .attr("title", function (d) { return d.description})
                            .text(textFunc)
                    });
            });
    }

    function updateWiring(wiringDiv, wiring) {
        return updateWiringUni(wiringDiv, wiring,
            function (w) {return bundleToPlugins[w.bundleId].pluginKey},
            function (w) {return w.value});
    }

    function updateDeclaredCapabilities(wiringDiv, wiring) {
        return updateWiringUni(wiringDiv, wiring,
            function (w) {return w.namespace},
            function (w) {return w.value});
    }

    function updateDeclaredRequirements(wiringDiv, wiring) {
        return updateWiringUni(wiringDiv, wiring,
            function (w) {return w.namespace},
            function (w) {return w.resolutionDirectives.filter});
    }

    function wiresByTypeToTypes(allWires)
    {
        var byType = {}, types = [];
        function add(resolutionType, d) {
            var wires = byType[resolutionType];
            if (!wires) {
                byType[resolutionType] = wires = [];
            }
            wires.push(d);
        }

        allWires.forEach(function(w){add(w.resolutionType, w)});
        Object.keys(byType).forEach(function(d){types.push({type: d, wires: byType[d]})});
        return types;
    }

    function groupBy(source, groupSelector)
    {
        var byGroup = [],
            map = {};

        function find(group) {
            var found = map[group];
            if (!found) {
                map[group] = found = {groupped:[], group:group};
                byGroup.push(found);
            }
            return found;
        }

        source.forEach(function(d){find(groupSelector(d)).groupped.push(d)});
        return byGroup;
    }

    // Lazily construct the package hierarchy from plugin keys.
    function packageHierarchy (bundles) {
        var map = {};

        function find(pluginKey, data) {
            var node = map[pluginKey], i;
            if (!node) {
                node = map[pluginKey] = data? {pluginKey: pluginKey, bundle: data} : {pluginKey: pluginKey, children: []};
                if (pluginKey.length) {
                    i = pluginKey.search(/\.[a-zA-Z0-9-]+[0-9.]*$/g);
                    node.parent = find(pluginKey.substring(0, i));
                    node.parent.children.push(node);
                }
            } else if (data) {
                // The node should be a leaf node - creating new and fixing up
                var leafNode = {pluginKey: pluginKey, bundle: data, parent: node};
                node.children.push(leafNode);
                node = leafNode;
            } else if (!node.children) {
                // The node should be an inner node - creating new and fixing up
                var leafNode = node,
                    siblings = leafNode.parent.children;

                node = leafNode.parent = map[pluginKey] = {
                    pluginKey: pluginKey,
                    children: [leafNode],
                    parent: leafNode.parent
                };

                // replace leafNode with node amongs siblings
                i = siblings.indexOf(leafNode);
                siblings.splice(i, 1);
                siblings.push(node);
            }
            return node;
        }

        bundles
            .forEach(function (d) {
                find(bundleToPlugins[d.bundleId].pluginKey, d);
            });

        return map[""];
    }

    // Return a list of dependencies for the given array of nodes.
    function pluginDeps (nodes, linkType) {
        var map = {},
            deps = [],
            resolutionTypes = {},
            leaves = nodes.filter(function(d){return d.bundle});

        var resolutionFilter = function(t){return linkType.indexOf(t) > -1};

        // Compute a map from bundleId to node.
        leaves.forEach(function (d) {
            map[d.bundle.bundleId] = d;
        });

        // For each import, construct a link from the source to target node.
        leaves.forEach(function (d) {
            var sourceBundleId = d.bundle.bundleId;

            d.bundle.dependencies.forEach(function (i) {
                var filteredTypes = i.resolutionTypes.filter(resolutionFilter);
                if (filteredTypes.length) {
                    deps.push({source: map[sourceBundleId], target: map[i.bundleId], targetBundleId: i.bundleId});
                    addTypes(sourceBundleId, i.bundleId, filteredTypes);
                }
            });

            if (resolutionFilter("service")) {
                var i, s = d.bundle.services.usingServicesFrom;
                for (i in s) {
                    if (sourceBundleId == s[i]) continue;
                    deps.push({source: map[sourceBundleId], target: map[s[i]], targetBundleId: s[i]});
                    addTypes(sourceBundleId, s[i], "service");
                }
            }
        });

        function addTypes(s,t, types) {
            var key = s+"->"+ t,
                list = resolutionTypes[key];
            if (!list) {
                resolutionTypes[key] = list = [];
            }
            list.push(types);
        }

        return {deps: deps, resolutionTypes: resolutionTypes};
    }

    function updateTab (event) {
        var match = event.target.id.match(/selector-deps-tab-(.+)/);
        if (match) {
            var section = match[1];

            if (tabHandlers[section]) {
                updateSection(section);
            }
        }
        return true;
    }

    function errorMessage(what, error) {
        AJS.messages.error({
            title: 'Error retrieving ' + what,
            body: '<p>'+error.statusText+'</p>',
            fadeout: true
        });
    }

    return module;
}(d3, AJS, (AJS && AJS.$) || $));
