package com.atlassian.depview.rest;

import com.atlassian.depview.osgi.BundleExplorer;
import com.google.common.collect.Maps;
import org.osgi.framework.Bundle;

import javax.xml.bind.annotation.XmlAccessType;
import javax.xml.bind.annotation.XmlAccessorType;
import javax.xml.bind.annotation.XmlAttribute;
import javax.xml.bind.annotation.XmlElement;
import java.net.URI;
import java.util.Arrays;
import java.util.Collection;
import java.util.Map;
import java.util.Optional;

@XmlAccessorType(XmlAccessType.FIELD)
public class BundleBean {
    @XmlAttribute
    private final Long bundleId;

    @XmlAttribute
    private final BundleState state;

    @XmlElement
    private final Collection<DependencyBean> dependencies;

    @XmlElement
    private final ServiceSummaryBean services;

    @XmlElement
    private final Map<String, URI> links;


    private enum BundleState {
        UNINSTALLED(Bundle.UNINSTALLED),
        INSTALLED(Bundle.INSTALLED),
        RESOLVED(Bundle.RESOLVED),
        STARTING(Bundle.STARTING),
        STOPPING(Bundle.STOPPING),
        ACTIVE(Bundle.ACTIVE);

        final private int bundleState;

        BundleState(final int bundleState) {
            this.bundleState = bundleState;
        }

        public static BundleState decode(final int state) {
            return Arrays.stream(BundleState.values())
                    .filter(s -> state == s.bundleState)
                    .findAny()
                    .orElseThrow(() -> new IllegalStateException("Unrecognised state [" + state + "]"));
        }
    }

    public BundleBean(final UriGenerator uriGenerator,
                      final Bundle bundle,
                      final Optional<String> query) {

        this.bundleId = bundle.getBundleId();
        this.state = BundleState.decode(bundle.getState());
        BundleExplorer bundleExplorer = new BundleExplorer(bundle, query);
        this.dependencies = bundleExplorer.getDependencies();
        this.services = new ServiceSummaryBean(bundleExplorer);

        links = Maps.newHashMap();
        links.put("self", uriGenerator.bundle(bundleId));

        Arrays.stream(new String[]{
                "providedWires",
                "requiredWires",
                "plugin",
                "servicesInUse",
                "registeredServices",
                "declaredCapabilities",
                "declaredRequirements"
        }).forEach(i -> links.put(i, uriGenerator.bundleInfo(bundleId, i)));
    }
}
