package com.atlassian.depview.osgi;

import com.atlassian.depview.rest.DependencyBean;
import com.atlassian.depview.rest.ServiceBean;
import com.atlassian.depview.rest.WireBean;
import com.google.common.collect.ImmutableSetMultimap;
import org.osgi.framework.Bundle;
import org.osgi.framework.Constants;
import org.osgi.framework.ServiceReference;
import org.osgi.framework.namespace.PackageNamespace;
import org.osgi.framework.wiring.BundleCapability;
import org.osgi.framework.wiring.BundleRevision;
import org.osgi.framework.wiring.BundleRevisions;

import javax.annotation.Nonnull;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.Map;
import java.util.Optional;
import java.util.stream.Collectors;
import java.util.stream.Stream;

public class BundleExplorer {
    private final Bundle bundle;
    private final Optional<String> query;

    public BundleExplorer(@Nonnull final Bundle bundle, Optional<String> query) {
        this.bundle = bundle;
        this.query = query;
    }

    public BundleExplorer(@Nonnull final Bundle bundle) {
        this(bundle, Optional.empty());
    }

    private String getResolutionType(final Map<String, String> resolutionDirectives) {
        return Optional.ofNullable(resolutionDirectives.get(PackageNamespace.REQUIREMENT_RESOLUTION_DIRECTIVE))
                .orElse(PackageNamespace.RESOLUTION_MANDATORY);
    }

    private Stream<BundleRevision> revisionStream() {
        return Optional.ofNullable(bundle.adapt(BundleRevisions.class))
                .map(BundleRevisions::getRevisions)
                .orElse(Collections.<BundleRevision>emptyList()).stream();
    }

    public Collection<WireBean> getDeclaredRequirements() {
        return revisionStream()
                .flatMap(br -> br.getDeclaredRequirements(null).stream())
                .map(req -> {
                    Map<String, String> resolutionDirectives = req.getDirectives();
                    final String namespace = req.getNamespace();
                    Object value = req.getAttributes().get(namespace);
                    return new WireBean(namespace, value, bundle.getBundleId(), resolutionDirectives, getResolutionType(resolutionDirectives), req.toString());
                })
                .collect(Collectors.toList());
    }

    public Collection<WireBean> getDeclaredCapabilities() {
        return revisionStream()
                .flatMap(br -> br.getDeclaredCapabilities(null).stream())
                .map(cap -> {
                    Map<String, String> resolutionDirectives = cap.getDirectives();
                    final String namespace = cap.getNamespace();
                    Object value = cap.getAttributes().get(namespace);
                    return new WireBean(namespace, value, bundle.getBundleId(), resolutionDirectives, getResolutionType(resolutionDirectives), cap.toString());
                })
                .collect(Collectors.toList());
    }

    public Collection<WireBean> getWires(WireDirection direction) {
        return revisionStream()
                .map(BundleRevision::getWiring)
                .filter(bw -> bw != null && bw.isInUse())
                .flatMap(bw -> direction.getWires(bw).stream())
                .map(wire -> {
                    final Long bundleId = direction.getBundle(wire).getBundleId();
                    Map<String, String> resolutionDirectives = wire.getRequirement().getDirectives();

                    final String namespace = wire.getRequirement().getNamespace();
                    Object value = null;
                    BundleCapability capability = wire.getCapability();
                    if (capability != null && capability.getAttributes().containsKey(namespace)) {
                        value = capability.getAttributes().get(namespace);
                    }

                    return new WireBean(namespace, value, bundleId, resolutionDirectives, getResolutionType(resolutionDirectives), wire.toString());
                })
                .filter(wb -> query.map(q -> wb.getNamespace().equals("osgi.wiring.package") && wb.getValue().toString().startsWith(q)).orElse(true))
                .collect(Collectors.toList());
    }

    public Collection<ServiceBean> getServices(WireDirection direction) {
        return Optional.ofNullable(direction.getServices(bundle))
                .map(r -> Arrays.stream(r)
                        .filter(this::filterServiceReference)
                        .map(ServiceBean::new)
                        .collect(Collectors.toList()))
                .orElse(Collections.emptyList());
    }

    private boolean filterServiceReference(final ServiceReference serviceReference) {
        return query
                .map(q -> {
                    String[] serviceInterfaces = (String[]) serviceReference.getProperty(Constants.OBJECTCLASS);
                    return Arrays.stream(serviceInterfaces)
                            .anyMatch(i -> i.startsWith(q));
                })
                .orElse(true);
    }

    public Collection<DependencyBean> getDependencies() {
        final ImmutableSetMultimap.Builder<Long, String> byBundleId = ImmutableSetMultimap.builder();

        getWires(WireDirection.REQUIRED).stream()
                .forEach(w -> {
                    byBundleId.put(w.getBundleId(), w.getResolutionType());
                });

        return byBundleId.build().asMap().entrySet().stream()
                .map(e -> new DependencyBean(e.getKey(), new HashSet<>(e.getValue())))
                .collect(Collectors.toList());
    }

}
