package com.atlassian.performance.seleniumjs

import org.openqa.selenium.By
import org.openqa.selenium.WebDriver
import org.openqa.selenium.support.ui.ExpectedCondition

class NativeExpectedConditions {
    companion object {
        @JvmStatic
        fun presenceOfElementLocated(locator: By): NativeExpectedCondition {
            return presenceOfElementLocated(LocatorConverters.toNativeBy(locator))
        }

        @JvmStatic
        fun visibilityOfElementLocated(locator: By): NativeExpectedCondition {
            return visibilityOfElementLocated(LocatorConverters.toNativeBy(locator))
        }

        @JvmStatic
        fun and(vararg conditions: NativeExpectedCondition): NativeExpectedCondition {
            return object : CompositeNativeExpectedCondition(conditions, "&&") {
                override fun toString(): String {
                    return "all conditions need to be valid: " + render()
                }
            }
        }

        @JvmStatic
        fun or(vararg conditions: NativeExpectedCondition): NativeExpectedCondition {
            return object : CompositeNativeExpectedCondition(conditions, "||") {
                override fun toString(): String {
                    return "at least one condition to be valid: " + render()
                }
            }
        }

        @JvmStatic
        fun seleniumOr(vararg conditions: NativeExpectedCondition): ExpectedCondition<Boolean> {
            return toSeleniumCondition(or(*conditions))
        }

        @JvmStatic
        fun seleniumAnd(vararg conditions: NativeExpectedCondition): ExpectedCondition<Boolean> {
            return toSeleniumCondition(and(*conditions))
        }

        @JvmStatic
        fun toSeleniumCondition(condition: NativeExpectedCondition): ExpectedCondition<Boolean> {
            return object : ExpectedCondition<Boolean> {
                override fun apply(driver: WebDriver?): Boolean {
                    return condition.apply(driver)
                }

                override fun toString(): String {
                    return condition.toString()
                }
            }
        }

        private fun presenceOfElementLocated(locator: NativeBy): NativeExpectedCondition {
            return object : NativeExpectedCondition() {
                override fun render(): String {
                    return locator.render()
                }

                override fun toString(): String {
                    return "presence of element located by: ${render()}"
                }
            }
        }

        private fun visibilityOfElementLocated(locator: NativeBy): NativeExpectedCondition {
            return object : NativeExpectedCondition() {
                override fun helperFunctions(): Set<String> {
                    return setOf(
                        """
function visibilityOfElement(element) {
	if (!element) {
		return false;
	}
	let style=window.getComputedStyle(element);
//    console.error("[" + element.getAttribute('id') + "] ["+style.display+"] ["+style.width+"] ["+style.height+"] [" + document.getElementById("key-val") + "]");
	return style.display !== 'none' && parseFloat(style.width)>0 && parseFloat(style.height)>0;
}
""".trimIndent()
                    )
                }

                override fun render(): String {
                    return "visibilityOfElement(${locator.render()})"
                }

                override fun toString(): String {
                    return "visibility of element located by: ${render()}"
                }
            }
        }
    }
}

abstract class NativeExpectedCondition {
    open fun helperFunctions(): Set<String> {
        return setOf()
    }

    fun apply(driver: WebDriver?): Boolean {
        val renderedHelperFunctions = if (helperFunctions().isEmpty()) "" else helperFunctions().joinToString("\n")

        val rendereredCondition = render()
//        val replace = rendereredCondition.replace('\'', '"')
        val scriptResults = JavaScriptUtils.executeScript<Boolean>(
            driver!!,
            "$renderedHelperFunctions\n"+
//                    "console.error('Evaluating: $replace');\n" +
                    "selenium_js_result=($rendereredCondition)\n" +
//                    "console.error('$replace => ' + !!selenium_js_result + ' [' + selenium_js_result + ']');\n" +
                    "return !!selenium_js_result"
        )
        return scriptResults
    }

    abstract fun render(): String
}

private open class CompositeNativeExpectedCondition(
    private val conditions: Array<out NativeExpectedCondition>,
    private val operator: String
) : NativeExpectedCondition() {
    override fun helperFunctions(): Set<String> {
        return conditions
            .map{ it.helperFunctions() }
            .flatten()
            .toCollection(LinkedHashSet())
    }

    override fun render(): String {
        return conditions
            .map{ "(" + it.render() + ")" }
            .joinToString(" $operator ")
    }
}
