package com.atlassian.paddle.connection;

import javax.naming.directory.DirContext;
import javax.naming.directory.InitialDirContext;
import javax.naming.Context;
import javax.naming.NamingException;
import java.util.Hashtable;

import com.atlassian.paddle.OutputReceiver;

public class DefaultConnectionFactory implements ConnectionFactory
{
    private Hashtable environment;
    private OutputReceiver outputReceiver;

    public DefaultConnectionFactory(ConnectionProperties properties, OutputReceiver outputReceiver)
    {
        this.environment = buildEnvironment(properties);
        this.outputReceiver = outputReceiver;
    }

    public DirContext createConnection() throws ConnectionException
    {
        String providerUrl = (String) environment.get(Context.PROVIDER_URL);
        try
        {
            long start = System.currentTimeMillis();
            InitialDirContext result = new InitialDirContext(environment);
            long tookMillis = System.currentTimeMillis() - start;
            outputReceiver.provideDebugFeedback("Connecting to LDAP server took " + tookMillis + " ms for " + providerUrl);
            return result;
        }
        catch (NamingException e)
        {
            throw createFriendlyLdapException(e, providerUrl);
        }
    }

    private ConnectionException createFriendlyLdapException(NamingException exception, String providerUrl)
    {
        String message = exception.getMessage();
        if (message != null && message.indexOf("error code 49") != -1)
        {
            return new ConnectionException("Username or password is incorrect. Please check them again.", exception);
        }

        Throwable cause = exception.getCause();
        if (cause != null && cause.getMessage() != null &&
            cause.getMessage().indexOf("Connection refused") != -1)
        {
            return new ConnectionException("Connection could not be opened to server at " + providerUrl + ". " +
                "Please check your host and port values are correct, and the server is running.", exception);
        }

        return new ConnectionException("Failed to connect to LDAP server with provider URL " +
            providerUrl + ": " + exception.getMessage(), exception);
    }

    private Hashtable buildEnvironment(ConnectionProperties properties)
    {
        Hashtable result = new Hashtable();
        result.put(Context.PROVIDER_URL, properties.getProviderUrl());
        result.put(Context.INITIAL_CONTEXT_FACTORY, "com.sun.jndi.ldap.LdapCtxFactory");
        result.put(Context.SECURITY_PROTOCOL, properties.getSecurityProtocol());
        result.put(Context.BATCHSIZE, properties.getBatchSize());

        if (!properties.isAnonymousAuthentication())
        {
            result.put(Context.SECURITY_PRINCIPAL, properties.getUsername());
            result.put(Context.SECURITY_CREDENTIALS, properties.getPassword());
        }
        result.put(Context.SECURITY_AUTHENTICATION, properties.getSecurityAuthentication());
        return result;
    }
}
