package com.atlassian.paddle.configuration;

import com.atlassian.paddle.connection.ConnectionProperties;
import com.atlassian.paddle.connection.DefaultConnectionProperties;
import com.atlassian.paddle.search.properties.*;
import com.atlassian.paddle.OutputReceiver;

public class AtlassianUserConfigurationReader implements ConfigurationReader
{
    private final Configuration configuration;
    private final OutputReceiver logger;

    /**
     * @param logger warnings will be written to this logger if a required configuration is missing.
     */
    public AtlassianUserConfigurationReader(Configuration configuration, OutputReceiver logger)
    {
        this.configuration = configuration;
        this.logger = logger;
    }

    public ConnectionProperties getConnectionProperties()
    {
        ConnectionProperties connectionProperties = new DefaultConnectionProperties(
            configuration.getAttributeInfoIfMissing("host", logger),
            configuration.getAttributeInfoIfMissing("port", logger),
            configuration.getAttributeInfoIfMissing("securityAuthentication", logger),
            configuration.getAttributeInfoIfMissing("securityProtocol", logger),
            configuration.getAttribute("securityPrincipal"),
            configuration.getAttribute("securityCredential"));

        if (connectionProperties.isAnonymousAuthentication())
        {
            logger.provideInfoFeedback("\nDetected that you are logging into LDAP anonymously.\n");
        }
        else
        {
            if (connectionProperties.getUsername() == null)
                logger.provideInfoFeedback("Username is missing from configuration, and anonymous authentication not configured");
            if (connectionProperties.getPassword() == null)
                logger.provideInfoFeedback("Password is missing from configuration, and anonymous authentication not configured");
        }
        return connectionProperties;
    }

    public UserSearchProperties getUserSearchProperties()
    {
        return new DefaultUserSearchProperties(
            configuration.getAttributeInfoIfMissing("baseUserNamespace", logger),
            configuration.getAttributeInfoIfMissing("userSearchFilter", logger),
            "true".equalsIgnoreCase(configuration.getAttribute("userSearchAllDepths")),
            configuration.getAttributeInfoIfMissing("usernameAttribute", logger),
            configuration.getAttributeInfoIfMissing("firstnameAttribute", logger),
            configuration.getAttributeInfoIfMissing("surnameAttribute", logger),
            configuration.getAttributeInfoIfMissing("emailAttribute", logger));
    }

    public GroupSearchProperties getGroupSearchProperties()
    {
        return new DefaultGroupSearchProperties(
            configuration.getAttributeInfoIfMissing("baseGroupNamespace", logger),
            configuration.getAttributeInfoIfMissing("groupSearchFilter", logger),
            "true".equalsIgnoreCase(configuration.getAttribute("groupSearchAllDepths")),
            configuration.getAttributeInfoIfMissing("groupnameAttribute", logger));
    }

    public MembershipSearchProperties getMembershipSearchProperties()
    {
        return new DefaultMembershipSearchProperties(
            configuration.getAttributeInfoIfMissing("membershipAttribute", logger),
            true,
            true);
    }
}
