package com.atlassian.paddle.configuration;

import org.dom4j.Document;
import org.dom4j.DocumentException;
import org.dom4j.Node;
import org.dom4j.io.SAXReader;

import java.io.*;

import com.atlassian.paddle.OutputReceiver;

/**
 * Parses an atlassian-user.xml file to retrieve requested LDAP properties.
 */
public class AtlassianUserConfiguration implements Configuration
{
    private Document atlassianUserXml;

    public AtlassianUserConfiguration(InputStream xml) throws ConfigurationException
    {
        setAtlassianUserXml(loadDocument(xml));
    }

    private Document loadDocument(InputStream xml) throws ConfigurationException
    {
        try
        {
            SAXReader reader = new SAXReader();
            return reader.read(xml);
        }
        catch (DocumentException e)
        {
            throw new ConfigurationException("Error parsing configuration XML: " + e.getMessage(), e);
        }
    }

    /**
     * Private setter used by constructors to encapsulate validation logic.
     */
    private void setAtlassianUserXml(Document document) throws ConfigurationException
    {
        if (!containsLdapNode(document))
        {
            throw new ConfigurationException("No LDAP settings found in XML configuration.");
        }

        atlassianUserXml = document;
    }

    /**
     * Returns <tt>true</tt> if the provided XML document contains LDAP configuration,
     * otherwise <tt>false</tt>.
     */
    private boolean containsLdapNode(Document document)
    {
        return document.selectSingleNode("//atlassian-user/repositories/ldap") != null;
    }

    public String getAttribute(String attributeName)
    {
        Node node = atlassianUserXml.selectSingleNode("//atlassian-user/repositories/ldap/" + attributeName);
        if (node == null) return null;
        return node.getStringValue();
    }

    public String getAttributeInfoIfMissing(String attributeName, OutputReceiver logger)
    {
        String value = getAttribute(attributeName);
        if (value == null)
            logger.provideInfoFeedback("Configuration does not have a setting for '" + attributeName + "'");
        return value;
    }
}
