package com.atlassian.paddle;

import com.atlassian.paddle.configuration.*;
import com.atlassian.paddle.connection.*;
import com.atlassian.paddle.task.*;
import com.atlassian.paddle.search.properties.*;
import org.apache.log4j.Logger;
import org.springframework.core.io.Resource;
import org.springframework.core.io.FileSystemResource;

import java.io.InputStream;
import java.io.IOException;
import java.util.Iterator;

public class Paddle
{
    private static final long BATCH_SIZE = 10;
    private static final int DEFAULT_COUNT_LIMIT = 10;
    private static final Logger log = Logger.getLogger(Paddle.class);
    private static final double version = 4.0;

    private final InputStream atlassianUserInputStream;

    private final Arguments arguments;

    private OutputReceiver outputReceiver;

    public Paddle(Arguments arguments, InputStream atlassianUserInputStream, OutputReceiver outputReceiver)
    {
        this.arguments = arguments;
        this.atlassianUserInputStream = atlassianUserInputStream;
        this.outputReceiver = outputReceiver;
    }

    public static void main(String[] args)
    {
        Arguments arguments = new Arguments(args);
        System.out.println("Note that this version no longer supplies a log4j.properties file! Please extract from the jar and rename");
        InputStream is;
        Resource xmlResource = new FileSystemResource("atlassian-user.xml");
        try
        {
            is = xmlResource.getInputStream();
        }
        catch (IOException e)
        {
            log.error("Error loading or finding atlassian-user.xml file: " + e.getMessage(), e);
            return;
        }

        LoggerOutputReceiver receiver = new LoggerOutputReceiver(log);
        Paddle paddle = new Paddle(arguments, is, receiver);
        paddle.testConnectivity();
    }

    public void testConnectivity()
    {
        if (arguments.contains("debug"))
        {
            outputReceiver.setDebuggingFeedback(true);
        }

        long countLimit = DEFAULT_COUNT_LIMIT;
        if (arguments.contains("limit"))
        {
            countLimit = Integer.parseInt(arguments.getValue("limit"));
        }

        outputReceiver.provideInfoFeedback("###########################################################################################################################");
        outputReceiver.provideInfoFeedback("");
        outputReceiver.provideInfoFeedback("LDAP Support Tool version " + version);
        outputReceiver.provideInfoFeedback("");
        outputReceiver.provideInfoFeedback("###########################################################################################################################\n");

        // try to load atlassian-user.xml file. expect it to be in the same directory
        ConfigurationReader configurationReader;
        try
        {
            Configuration configuration = new AtlassianUserConfiguration(atlassianUserInputStream);
            configurationReader = new AtlassianUserConfigurationReader(configuration, outputReceiver);
        }
        catch (ConfigurationException e)
        {
            outputReceiver.provideErrorFeedback("Error reading atlassian-user.xml file: " + e.getMessage(), e);
            return;
        }

        ConnectionProperties connectionProperties = configurationReader.getConnectionProperties();
        UserSearchProperties userSearchProperties = configurationReader.getUserSearchProperties();
        MembershipSearchProperties membershipSearchProperties = configurationReader.getMembershipSearchProperties();
        GroupSearchProperties groupSearchProperties = configurationReader.getGroupSearchProperties();

        ConnectionFactory connectionFactory = new DefaultConnectionFactory(connectionProperties, outputReceiver);
        TaskRunner taskRunner = new DefaultTaskRunner(connectionFactory);

        try
        {
            taskRunner.runTask(new ConnectionTestTask(outputReceiver), countLimit);
        }
        catch (ConnectionException e)
        {
            outputReceiver.provideErrorFeedback("Failed to connect to LDAP server: " + e.getMessage(), e);
            return;
        }

        outputReceiver.provideInfoFeedback("-----------------------------------------------------------------");
        outputReceiver.provideInfoFeedback("TEST 1: Search and list " + countLimit + " users");
        outputReceiver.provideInfoFeedback("-----------------------------------------------------------------\n");

        try
        {
            // try to query for some users
            ListUsersTask task = new ListUsersTask(userSearchProperties, membershipSearchProperties,
                    groupSearchProperties, outputReceiver);
            taskRunner.runTask(task, countLimit);
        }
        catch (ConnectionException e)
        {
            // connection test should have failed if we can't connect, so bail out now
            throw new RuntimeException(e);
        }

        outputReceiver.provideInfoFeedback("-----------------------------------------------------------------");
        outputReceiver.provideInfoFeedback("TEST 2: Search and list " + countLimit + " groups");
        outputReceiver.provideInfoFeedback("-----------------------------------------------------------------\n");

        try
        {
            // try to query for some groups
            ListGroupsTask task = new ListGroupsTask(groupSearchProperties, membershipSearchProperties, outputReceiver);
            taskRunner.runTask(task, countLimit);
        }
        catch (ConnectionException e)
        {
            // connection test should have failed if we can't connect, so bail out now
            throw new RuntimeException(e);
        }


        if (taskRunner.hasErrors())
        {
            outputReceiver.provideInfoFeedback("The following errors were encountered in running LDAP tests:");
            for (Iterator iterator = taskRunner.getErrorsIterator(); iterator.hasNext();)
            {
                String errorMessage = (String) iterator.next();
                outputReceiver.provideInfoFeedback("");
                outputReceiver.provideInfoFeedback(errorMessage);
            }
            outputReceiver.provideInfoFeedback("");
            outputReceiver.provideInfoFeedback("Please submit a support ticket at http://support.atlassian.com and attach your atlassian-user.xml and the paddle.log file (located in this directory).");
        }
    }
}
