package com.atlassian.oauth2.client.api.storage;

import com.atlassian.annotations.PublicApi;
import com.atlassian.oauth2.client.api.ClientToken;
import com.atlassian.oauth2.client.api.ClientTokenMetadata;
import com.atlassian.oauth2.client.api.storage.token.exception.RecoverableTokenException;
import com.atlassian.oauth2.client.api.storage.token.exception.UnrecoverableTokenException;

import java.time.Duration;

/**
 * Utilities for working with OAuth 2.0 tokens.
 */
@PublicApi
public interface TokenHandler {
    /**
     * Handles the use of {@link ClientToken}, automatically applying a refresh before executing the callback if
     * required.
     * <p>
     * This method may update the {@link ClientTokenMetadata} of the token.
     * <p>
     * Note that the {@link ClientTokenCallback} may be executed more than once if it throws
     * {@link InvalidTokenException}.
     *
     * @param clientTokenId the id of the token to be used in the execution, or to be refreshed if required
     * @param callback      the callback to be executed.
     * @param <T>           the result type.
     * @return the result of the callback
     * @throws UnrecoverableTokenException   thrown if the token is invalid and can't be automatically refreshed
     * @throws RecoverableTokenException thrown if the token is invalid but it might be possible to recover
     */
    <T> T execute(String clientTokenId, ClientTokenCallback<T> callback) throws UnrecoverableTokenException, RecoverableTokenException;

    /**
     * Handles the use of {@link ClientToken}, automatically applying a refresh before executing the callback if
     * required.
     * <p>
     * This method may update the {@link ClientTokenMetadata} of the token.
     * <p>
     * Note that the {@link ClientTokenCallback} may be executed more than once if it throws
     * {@link InvalidTokenException}.
     *
     * @param clientTokenId the id of the token to be used in the execution, or to be refreshed if required
     * @param callback      the callback to be executed.
     * @param margin        only refreshes token if expiry is before now + margin
     * @param <T>           the result type.
     * @return the result of the callback
     * @throws UnrecoverableTokenException   thrown if the token is invalid and can't be automatically refreshed
     * @throws RecoverableTokenException thrown if the token is invalid but it might be possible to recover
     */
    <T> T execute(String clientTokenId, ClientTokenCallback<T> callback, Duration margin) throws UnrecoverableTokenException, RecoverableTokenException;

    /**
     * Refreshes and returns the client token, saving the up to date token to the store. May update
     * {@link ClientTokenMetadata} of the token.
     *
     * @param clientTokenId the id of the token to be refreshed
     * @return the refreshed token
     * @throws UnrecoverableTokenException   thrown if the token is invalid and can't be automatically refreshed
     * @throws RecoverableTokenException thrown if the token is invalid but it might be possible to recover
     */
    ClientToken getRefreshedToken(String clientTokenId) throws UnrecoverableTokenException, RecoverableTokenException;

    /**
     * Refreshes if access token is already expired or expires within the margin period. Returns the client token,
     * saving the up to date token to the store.
     * If the refreshed token lifetime is shorter than the {@code margin} then the refreshed token is returned. It means
     * that in such case returned token will be valid only for part of the {@code margin} period.
     * May update {@link ClientTokenMetadata} of the token.
     * @param clientTokenId the id of the token to be refreshed if required
     * @param margin only refreshes token if expiry is before now + margin
     * @return the token, potentially refreshed
     * @throws UnrecoverableTokenException   thrown if the token is invalid and can't be automatically refreshed
     * @throws RecoverableTokenException thrown if the token is invalid but it might be possible to recover
     */
    ClientToken getRefreshedToken(String clientTokenId, Duration margin) throws UnrecoverableTokenException, RecoverableTokenException;

    /**
     * Callback to execute with a {@link ClientToken}.
     *
     * @param <T> returned type
     */
    @FunctionalInterface
    interface ClientTokenCallback<T> {
        /**
         * Method to invoke with a {@link ClientToken}.
         * <p>
         * If authorisation with the token fails, this method should throw {@link InvalidTokenException}.
         *
         * @param clientToken token to be used for authorisation with an OAuth 2.0 server
         * @return outcome of the invocation
         * @throws InvalidTokenException thrown if the token is invalid and authorisation failed
         */
        T apply(ClientToken clientToken) throws InvalidTokenException;
    }

    /**
     * Indicates that there was a problem with a {@link ClientToken} and it couldn't be used for successful
     * authorisation with an OAuth 2.0 resource server.
     */
    class InvalidTokenException extends Exception {
        public InvalidTokenException() {
        }

        public InvalidTokenException(String message) {
            super(message);
        }

        public InvalidTokenException(String message, Throwable cause) {
            super(message, cause);
        }

        public InvalidTokenException(Throwable cause) {
            super(cause);
        }
    }
}
