AJS.EventQueue = AJS.EventQueue || [];

/**
 * A model to represent the "group" information for notifications that will be aggregated
 * Model should hold information common to notifications that it is associated with
 */
MW.NotificationGroup = Backbone.Model.extend({

    initialize: function(group, options){
        var globalEvents = this.globalEvents = options.globalEvents;
        this.config = options.config;
        var that = this;

        // Create notifications associated with this notification group
        this.notifications = [];

        // Note: Ideally would want to do the binding on set
        // but 'validate' (our only nice-ish way to override set) seems
        // to be called multiple times :(
        if(globalEvents) {
            globalEvents.on("focusClickedNotification", this.focusSelected, this);
            globalEvents.on("removeDrilldownFocused", function() {
                if (that.get("drilldownFocused")) {
                    that.set("drilldownFocused", false);
                }
            });
            globalEvents.on("removeMainFocused", function() {
                if (that.get("mainFocused")) {
                    that.set("mainFocused", false);
                }
            });
        }
    },

    setState: function(status, callback) {
        var that = this;
        this.set("status", status);
        // TODO should be changed to groupingId (instead of relying on the first notification)
        var notificationId = this.notifications[0].get("id");
        MW.$.ajax({
            url:  MW.contextPath + '/rest/mywork/latest/notification/' + notificationId + '/status',
            type: "PUT",
            contentType: "application/json",
            data: JSON.stringify(status),
            success: function(result) {
                callback && callback();
                if (!_.isObject(result)) {
                    // Zepto returns string and not JSON object!
                    result = JSON.parse(result);
                }
                // The previous task my still exist but be completed - blank out the globalId
                MW.Tasks.getByGlobalIdDeferred(result.globalId, function(task) {
                    if (task && task.isCompleted()) {
                        task.set('globalId', '', {silent: true});
                    }
                });
                MW.Tasks.add(result, {at: 0});
            }
        });

    },

    hasTask: function() {
        return this.get("status") === 'TODO';
    },

    createTask: function(callback) {
        this.setState("TODO", callback);

        var globalId = this.get('globalId');
        MW.Notifications.setStatusByGlobalId(globalId, "TODO");

        this.globalEvents.trigger("taskCreated");

        AJS.EventQueue.push({name:'mywork.createTask', properties:{
            'changeStatusTo': this.get("status")
        }});
    },

    appendNotifications: function(notifications) {
        var that = this;
        notifications = _.map(notifications, function(notification) {
            var attributes = MW.$.extend(notification, {globalEvents: that.globalEvents, parent: that});
            return new MW.Notification(attributes);
        });
        this.notifications = _.sortBy(notifications.concat(this.notifications), function(notification) {
            return notification.get('created') * -1;
        });

        this.updateStatus();
        this.set('notifications', this.notifications);

        this.set('updated', _.first(this.notifications).get('created'));
        this.trigger("modified", this);

        this.globalEvents.trigger("removeDrilldownFocused");
    },

    getPrettyTime: function() {
        return MW.getPrettyTime(_.first(this.notifications).get('created'));
    },

    getUnreadNotifications: function() {
        return _.filter(this.get('notifications'), function(notification) {
            return !notification.get('read');
        });
    },

    getMetadata: function() {
        return this.notifications[0].get("metadata");
    },

    resetFocused: function(notification) {
        this.removeFocus();
        (notification || _.first(this.notifications)).set("drilldownFocused", true, {silent: true});
    },

    updateStatus: function() {
        // Status is only relevant when all notifications in the group have the same globalId
        var globalId = this.notifications[0].get('globalId');
        if (!_.any(this.notifications, function (notification)
        {
            return notification.get('globalId') !== globalId;
        }))
        {
            this.set("globalId", this.notifications[0].get("globalId"));
            this.set("status", this.notifications[0].get("status"));
        }
        else
        {
            this.unset("globalId");
            this.unset("status");
        }
    },

    setStatusByGlobalId: function(globalId, status) {
        if (this.get("globalId") === globalId) {
            this.set("status", status);
            _.each(this.notifications, function(notification) {
                notification.set('status', status);
            });
        } else {
            _.each(this.notifications, function(notification) {
                if (notification.get("globalId") === globalId) {
                    notification.set('status', status);
                }
            });
        }
    },

    hasNewNotifications: function() {
        return !(this.notifications)[0].get("read");
    },

    /**
     * Plural version of i18n.
     * Assumes list of i18n arguments rather than map.
     */
    i18ns: function(extra, number) {
        var plural = number > 1 ? "s" : "";
        return this.i18n.apply(this, [ extra + plural ].concat(Array.prototype.slice.call(arguments, 1)));
    },

    /**
     * Either pass a single object or an array of _primitive_ arguments.
     */
    i18n: function(extra) {
        var map = {};
        _.each(Array.prototype.slice.call(arguments, 1), function(arg, i) {
            map[i] = arg;
        });
        return this.i18nWithMap(extra, map);
    },

    i18nWithMap: function(extra, map) {
        return this.entityI18n(this.get('action') + (extra ? "." + extra : ""), map);
    },

    i18nHtml: function(extra, map) {
        var html = MW.escapeHTML(this.i18nWithMap(extra, MW.$.extend({user: '{user}'}, map)));
        if (!html) {
            return "";
        }
        var metadata = this.getMetadata();
        var href = this.config.getUserUrl(this.get('application'), metadata.username);
        var user = metadata.user || "";
        var userLink = href ? MW.$("<a />").attr("target", "_blank").attr("href", href).text(user).wrap("<div />").parent().html() : MW.escapeHTML(user);
        return html.replace('{user}', userLink);
    },

    entityI18n: function(extra, map) {
        var key = this.get('category') + (extra ? "." + extra : ""),
            value = this.config.i18n(key);
        if (!value) {
            return ""; // Allow for defaults with ||
        }
        return MW.formatString(value, _.extend({}, this.getMetadata(), map));
    },

    focused: function() {
        for (var i = 0; i < this.notifications.length; i++) {
            if (!!this.notifications[i].get("drilldownFocused")) {
                return this.notifications[i];
            }
        }
    },

    focusOlder: function() {
        var currIndex = _.indexOf(this.notifications, this.focused());

        var next = this.notifications[Math.min(this.notifications.length - 1, currIndex + 1)];
        if (!next.get('drilldownFocused')) {
            this.removeFocus();
            next.set("drilldownFocused", true);
        }
        return next;
    },

    focusNewer: function() {
        var currIndex = _.indexOf(this.notifications, this.focused());

        if (currIndex > 0) {
            this.removeFocus();
            this.notifications[currIndex - 1].set("drilldownFocused", true);
        }
    },

    focusSelected: function(notification) {
        var currFocusedNotification = this.focused();
        if ( currFocusedNotification && currFocusedNotification !== notification) {
            this.removeFocus();
            notification.set("drilldownFocused", true);
        }
    },

    focusOldest: function() {
        // Focus the oldest notification (used when getting focus on the top notification when expanding all notifications)
        this.removeFocus();
        _.last(this.notifications).set("drilldownFocused", true);
    },

    removeFocus: function() {
        _.each(this.notifications, function(notification) {
            notification.set("drilldownFocused", false);
        });
    }

});