/**
 * Given a model describing an action, renders the list element for that action
 * and binds appropriate events to handle forms.
 */
MW.ActionAbstractView = Backbone.View.extend({

    tagName: "li",
    className: "action-item",
    disabled: false,

    events: function() {
        return {
            "click": "clickEvent"
        };
    },

    disableFormElements: function() {
        var $formElement = this.options.actionFormElement;
        $formElement.find("input").addClass("disabled").attr("disabled", "disabled");
    },

    enableFormElements: function() {
        var $formElement = this.options.actionFormElement;
        $formElement.find("input").removeClass("disabled").removeAttr("disabled");
    },

    showSuccessMessage: function(data) {
        var attrs = this.model.attrs || {};

        if (attrs.alternateAction) {
            var oldAction = this.model;
            this.model = attrs.getAlternateAction();
            this.options.actions[_.indexOf(this.options.actions, oldAction)] = this.model;
            this.render();
            return;
        }

        if (attrs.allowOnceOnly) {
            this.disableActionElement();
        }
        var alternateDisplayName = this.actionI18n('alternateDisplayName');
        if (alternateDisplayName) {
            this.toggleDisplayName(this.actionI18n('displayName'), alternateDisplayName, attrs.responseLink);
        }
        if (attrs.responseLink) {
            this.$el.unbind().find("a").attr("target", "_blank").attr("href", data);
        }
    },

    makeAjaxRequest: function(successCallback, errorCallback, extraData) {
        var that = this,
            attrs = this.model.attrs || {},
            $formElement = this.options.actionFormElement,
            actionMessageView = that.options.actionMessageView;

        this.disableActionElement();
        $formElement.find('.aui-button-primary').addClass('disabled');
        $formElement.find('.mw-loading').addClass('show');
        var stopDots = this.addDots();
        var finished = function() {
            stopDots();
            that.enableActionElement();
            $formElement.find('.aui-button-primary').removeClass('disabled');
            $formElement.find('.mw-loading').removeClass('show');
        };
        var success = function(data) {
            finished();
            actionMessageView.clearMessage();
            $formElement.parent().removeClass('expanded');
            successCallback ? successCallback(data) : that.showSuccessMessage(data);
        };

        this.options.notification.executeInlineAction({
            actionName: this.model.id,
            target: this.options.target,
            extraData: extraData,
            success: function(result) {
                if (result.successful) {
                    if (!attrs.instant) {
                        success();
                    }
                } else {
                    finished();
                    var msg = that.i18n(result.messageKey) || result.message || AJS.I18n.getText('com.atlassian.mywork.action.unknown.error');
                    if (errorCallback) {
                        errorCallback(msg, null, result.authUrl);
                    } else {
                        if (result.authUrl) {
                            actionMessageView.showOAuthErrorMessage(AJS.I18n.getText('com.atlassian.mywork.action.oauth.required'), result.authUrl);
                        } else {
                            actionMessageView.showErrorMessage(msg);
                        }
                    }
                }
            },
            error: function(xhr, textStatus, message) {
                finished();
                if (errorCallback) {
                    errorCallback(message, xhr);
                } else {
                    if (xhr.status === 401) {
                        actionMessageView.showLoginErrorMessage(AJS.I18n.getText('com.atlassian.mywork.action.unauthorised'));
                    } else {
                        actionMessageView.showErrorMessage(message);
                    }
                }
            }
        });
        if (attrs.instant) {
            success();
        }
    },

    addDots: function() {
        var $actionLink = MW.$(this.el).find('a');
        var text = $actionLink.text();
        $actionLink.text(text + "...");
        return function() {
            $actionLink.text(text);
        };
    },

    clickEvent: function(e) {
        var type = this.model.type,
            that = this;

        if (this.disabled) {
            return false;
        }

        if (type !== "link" && type !== "taskLink") {
            e.preventDefault();
        }

        if (type === "ajax") {
            this.makeAjaxRequest(null, null, null);
        }

        if (type === "form") {
            // The element in which the form should be placed.
            var $formElement = this.options.actionFormElement;
            // Remove any existing forms first.
            $formElement.find("form").remove();
            $formElement.append(this.renderForm());
            $formElement.parent().addClass('expanded');
            this.showForm($formElement);
        }
    },

    toggleDisplayName: function(displayName, alternateDisplayName, keepLink) {
        var link;
        if (!!keepLink) {
            link = MW.$(this.el).find("a");
        } else {
            link = MW.$(this.el);
            link.unbind();
        }
        var linkText = (link.text() === displayName) ? alternateDisplayName : displayName;
        link.text(linkText);
    },

    disableActionElement: function() {
        // If the user should only be allowed to perform this action once,
        // disable the action link.
        this.setDisabled(true);
    },

    enableActionElement: function() {
        this.setDisabled(false);
    },

    setDisabled: function(disabled) {
        this.disabled = disabled;
        MW.$(this.el).toggleClass("disabled", disabled);
    },

    /**
     * Completely remove the form element (specified in the options as actionFormElement)
     */
    removeForm: function() {
        var $formElement = this.options.actionFormElement;

        // If the element is still visible, slide up before removing completely.
        if ($formElement.visible()) {
            // TODO: implement as slideUp for desktop
//            $formElement.slideUp('fast');
            $formElement.find('textarea').blur();
            $formElement.remove();
            this.enableActionElement();
        }
    },

    doFormBindings: function($form) {
        var that = this;

        // Prevent keypresses from activating keyboard shortcuts.
        $form.keypress(function(e) {
            e.stopPropagation();
        });

        var updateSubmitButton = function() {
            var submitButton = $form.find(".aui-button-primary");
            var textarea = $form.find("textarea");
            var hasText = textarea.val().length > 0;
            submitButton.toggleClass("disabled", !hasText);
            if (hasText) {
                submitButton.removeAttr("disabled");
            } else {
                submitButton.attr("disabled", "disabled");
            }
        };
        $form.keyup(updateSubmitButton);
        updateSubmitButton();

        // Don't trigger a page refresh on cancel, just hide the form.
        $form.find(".inline-cancel").click(function(e) {
            e.preventDefault();
            that.removeForm();
        });

        // On submit, do an ajax request using the previously defined params.
        $form.submit(function(e) {
            that.disableFormElements();

            var $serializedForm = $form.serializeObject();

            $form.find('textarea').blur().attr('disabled', 'disabled');
            that.makeAjaxRequest(function(data) {
                that.showSuccessMessage(data);
                that.removeForm();
            }, function(message, xhr, authUrl) {
                var actionMessageView = that.options.actionMessageView;
                $form.find('textarea').removeAttr('disabled');
                if (xhr && xhr.status === 401) {
                    actionMessageView.showLoginErrorMessage(AJS.I18n.getText('com.atlassian.mywork.action.unauthorised.form'));
                } else if (authUrl) {
                    actionMessageView.showOAuthErrorMessage(AJS.I18n.getText('com.atlassian.mywork.action.oauth.required.form'), authUrl);
                } else {
                    actionMessageView.showErrorMessage(message);
                }
                that.enableFormElements();
            }, $serializedForm);
            return false;

        });
    },

    renderForm: function() {
        // Don't let the User click the Action link again and blow away their form content. That's rude.
        this.disableActionElement();

        var that = this,
            form = this.getFormContainerTemplate(),
            $form = MW.$(form);

        // Loop through each of the form elements and add them to the form.
        _.each(this.options.model.data, function(formElement) {
            var element;

            if (formElement.type === "textarea") {
                element = that.getFormTextareaTemplate();
            }

            $form.append(element);
        });

        // Add on the action buttons to the end of forms (eg submit, cancel etc).
        // May want to make these optional in future.
        var actionButtons = this.getFormActionButtonsTemplate({submitLabel:this.actionI18n('submitLabel')});
        $form.append(actionButtons);
        this.doFormBindings($form);

        return $form;
    },

    render: function() {
        var $el = MW.$(this.el);

        $el.empty();

        if (this.model.type === "link") {
            var notificationModel = this.options.notification;
            var notification = notificationModel.toJSON();
            var url = this.options.url;
            if (this.model.url) {
                var appLink = MW.Configurations.getFromNotification(notificationModel.attributes);
                var baseUrl = appLink ? appLink.get('url') : MW.contextPath;
                url = baseUrl + MW.formatString(this.model.url, Mw.$.extend({globalId: notification.globalId}, notification.metadata));
            }
            var linkText = this.actionI18n('displayName') || AJS.I18n.getText("com.atlassian.mywork.action.open");
            var $link = MW.$(MyWork.Templates.openLinkAction({notification: notification, url: url, displayName: linkText}));
            $link.mouseup(function() {
                notificationModel.get('parent').trigger("open", notificationModel);
            });
            $el.append($link);
        } else if (this.model.type === "taskLink") {
            $el.append(MW.$("<a>")
                .attr("href", this.options.url)
                .attr("target", "_blank")
                .text(this.actionI18n('displayName'))
            );
        } else {
            $el.append(MW.$("<a href='#'>" + this.actionI18n('displayName') + "</a>"));
        }

        if (this.actionI18n('title')) {
            $el.attr("title", this.actionI18n('title'));
        }

        return this;
    },

    actionI18n: function(field) {
        var actionId = this.model.id;

        return this.i18n('action.' + actionId + '.' + field);
    },

    i18n: function(extra) {
        var notification = this.options.notification.toJSON(),
            config = notification.parent.config,
            application = notification.parent.get('application');

        return config.i18n(application + '.' + extra);
    }
});
