MW.NotificationCollection = Backbone.Collection.extend({
    currentLastFetchId: 0,
    beforeId: 10000000,

    fetcher: function(options) {
        options = options || {};
        if (!this.firstFetch && options.updateOnly) {
            return;
        }
        if(!this.firstFetch) {
            this.firstFetch = true;
            return this.fetchWithLimit(this.hasFetched.resolveOnSuccess(options));
        } else {
            return this.fetchAfterWithLimit(options);
        }
    },

    fetchWithLimit: function(options) {
        options.data = options.data || {};
        options.data = _.extend(options.data, {
            limit: MW.MAX_RESULTS
        });
        return this.fetch(options);
    },

    // A smart fetch which adds and fetches after currentLastFetchId
    fetchAfterWithLimit: function(options){
        options = _.extend(options, {add: true});
        options.data = options.data || {};
        options.data = _.extend(options.data, {
            after: this.currentLastFetchId
        });
        return this.fetchWithLimit(options);
    },

    initialize: function(model, options) {
        var that = this;
        this.globalEvents = options.globalEvents;
        this.url = MW.contextPath + "/rest/mywork/latest/notification/nested";

        this.model = MW.NotificationGroup;

        this.hasFetched = MW.Deferred();

        this.config = options.config;
    },

    comparator: function(g1, g2) {
        var pinned1 = g1.attributes.pinned,
            pinned2 = g2.attributes.pinned;
        if (pinned1 && !pinned2) return -1;
        if (!pinned1 && pinned2) return 1;
        var updated1 = g1.attributes.updated,
            updated2 = g2.attributes.updated;
        if (updated1 < updated2) return 1;
        if (updated1 > updated2) return -1;
        return 0;
    },

    setLastReadId: function(lastReadId) {
        if (lastReadId <= this.currentLastFetchId) {
            return;
        }
        this.currentLastFetchId = lastReadId;
        MW.$.ajax({
                url: MW.contextPath + "/rest/mywork/latest/notification/lastreadid",
                type: "PUT",
                contentType: "application/json",
                data: JSON.stringify(this.currentLastFetchId)
        });
    },

    getCategory: function(notificationGroup) {
        return notificationGroup.application + "." + notificationGroup.entity;
    },

    geti18nKey: function(notificationGroup) {
        return this.getCategory(notificationGroup) + "." + notificationGroup.action;
    },

    isMyWorkAuth: function(notificationGroup) {
        return notificationGroup.application == 'com.atlassian.mywork.host.provider' && notificationGroup.entity == 'authentication';
    },

    prepareMyWorkAuth:function (notificationGroup, notifications, baseUrl) {
        var callback = MW.createCallback(baseUrl);

        if (callback) {
            _.each(notifications, function (notification) {
                notification.url += MW.appendCallback(notification.url, callback);
                notification.target = "_top";
            });
            notificationGroup.url = notifications[0].url;
            notificationGroup.target = "_top";
        }
    },

    parse: function(response) {
        var that = this;

        var notifications = [];
        _.each(response, function(group) {
            var notificationGroup = that.initNotification(group);
            notifications = notifications.concat(notificationGroup);
        });

        if (!this.currentLastFetchId && !!notifications.length) {
            notifications[0].mainFocused = true;
        }

        var fetchId = 0;
        var newGroups = [];
        _(notifications).each(function(notificationGroup){
            _(notificationGroup.notifications).each(function(notification) {
                if (notification.id > fetchId) {
                    fetchId = notification.id;
                }
                that.beforeId = Math.min(that.beforeId, notification.id);
            });

            var previousGroup = that.getByAggregateKey(notificationGroup.aggregateKey);
            if (previousGroup) {
                return previousGroup.appendNotifications(notificationGroup.notifications);
            }

            var combinedGroup = new MW.NotificationGroup(notificationGroup, {
                globalEvents: that.globalEvents,
                config: that.config.getFromNotification(notificationGroup)
            });
            combinedGroup.appendNotifications(notificationGroup.notifications);
            newGroups.push(combinedGroup);
        });

        this.setLastReadId(fetchId || this.currentLastFetchId);

        // Only return new groups
        return newGroups;
    },

    initNotification: function(group) {
        var that = this,
            notificationGroup = group.item,
            notifications = group.notifications,
            hasConfig = this.config.hasConfig(notificationGroup),
            notificationConfig = this.config.getFromNotification(notificationGroup),
            i18nKey = this.geti18nKey(notificationGroup);

        if (this.isBasicAction(notificationGroup)) {
            notificationGroup.displayIconClass = "mw-icon-" + this.getIconClassName(notificationGroup);
        }

        notificationGroup.category = that.getCategory(notificationGroup);

        if (hasConfig) {
            notificationGroup.objectActions = notificationConfig.getObjectActions(notificationGroup.entity, notificationGroup.action, notifications[0].metadata);
            if (notificationGroup.url && notificationGroup.url.indexOf('/') === 0) {
                notificationGroup.url = notificationConfig.get('url') + notificationGroup.url;
            }
            if (notificationGroup.iconUrl && notificationGroup.iconUrl.indexOf('/') === 0) {
                notificationGroup.iconUrl = notificationConfig.get('url') + notificationGroup.iconUrl;
            }
        } else if (notificationGroup.url) {// Header with URL but without config should have 'Open' action
            notificationGroup.objectActions = [{type: "link"}];
        } else {
            notificationGroup.objectActions = [];
        }
        notificationGroup.notifications = [];

        if (this.isMyWorkAuth(notificationGroup)) {
            this.prepareMyWorkAuth(notificationGroup, notifications, MW.getBaseUrl());
        }

        _.each(notifications, function(notification) {

            // TODO go through and check/cleanup old/uneeded attributes
            notification.target = notification.target || "_blank";

            notification.descriptionHtml = (notification.description || "");
            notification.highlightTextHtml = notification.metadata && notification.metadata.highlightText;

            var hasTags = _(["br", "ul", "p"]).chain().map(function(tag) {
                return notification.descriptionHtml.indexOf("<" + tag) >= 0;
            }).any().value();

            if (!hasTags) {
                notification.descriptionHtml = notification.descriptionHtml.replace(/\n/g, "<br/>");
            }

            if (hasConfig) {
                notification.actions = notificationConfig.getActions(notificationGroup.entity, notificationGroup.action, notification.metadata);
                if (notification.url && notification.url.indexOf('/') === 0) {
                    notification.url = notificationConfig.get('url') + notification.url;
                }
                if (notification.iconUrl && notification.iconUrl.indexOf('/') === 0) {
                    notification.iconUrl = notificationConfig.get('url') + notification.iconUrl;
                }
                if (notification.actionIconUrl && notification.actionIconUrl.indexOf('/') === 0) {
                    notification.actionIconUrl = notificationConfig.get('url') + notification.actionIconUrl;
                }
            } else if (notification.url) { // Notifications with URL but without config should have 'Open' action
                notification.actions = [{type: "link"}];
            } else {
                notification.actions = [];
            }

            notificationGroup.notifications.push(notification);
        });

        // Due to relative baseURLs - assign displayIconUrl as the last thing (we're potentially calculating the proper URL above)
        notificationGroup.displayIconUrl = notifications[0].actionIconUrl || notificationGroup.iconUrl || notificationConfig.appendUrl(i18nKey) || notificationConfig.appendUrl(notificationGroup.category);

        return notificationGroup;
    },

    // If it's a "basic action" (ie. we have icon for it) then we can use css-class avoid having inline background style
    isBasicAction: function(notificationGroup) {
        // If it's a JIRA issue, we've only got the comment icon (don't have all the issue icons!)
        if (notificationGroup.application === "com.atlassian.mywork.providers.jira") {
            return notificationGroup.action === "comment";
        } else if (notificationGroup.application === "com.atlassian.mywork.providers.confluence") {
            return _.include(["like", "comment", "task.assign", "mentions.user", "share", "resolve", "reopen", "invite"], notificationGroup.action);
        }
        return false;
    },

    getIconClassName: function(notificationGroup) {
        var className = notificationGroup.action;

        if (className.indexOf("task") !== -1) {
            className = "inline-task";
        }

        if (_.include(["mentions.user", "share", "invite"], className)) {
            className = notificationGroup.entity;
        }

        return className;
    },

    /**
     * This returns the notifications that should appear in the main page.
     */
    main: function() {
        return this.models;
    },

    unread: function() {
        return this.filter(function(notification) {
            return !notification.get("read");
        });
    },

    setNotificationsAsRead: function(notifications, isSilent) {
        var ids = _(notifications).chain().filter(function(notification) {
            return !notification.get('read');
        }).map(function(notification) {
            notification.set("read", true, {silent: isSilent});
            return notification.id;
        }).value();

        if (ids.length > 0) {
            MW.$.ajax({
                url: MW.contextPath + "/rest/mywork/latest/notification/read",
                type: "PUT",
                contentType: "application/json",
                data: JSON.stringify(ids)
            });
        }
    },

    getByAggregateKeyDeferred: function(key, callback) {
        var that = this;
        this.hasFetched.done(function(){
            callback(that.getByAggregateKey(key));
        });
    },

    getByAggregateKey: function(key) {
        return this.find(function(notificationGroup){
            return notificationGroup.get("aggregateKey") === key;
        });
    },

    setStatusByGlobalId: function(globalId, status)
    {
        this.each(function(notificationGroup) {
            notificationGroup.setStatusByGlobalId(globalId, status);
        });
    },

    drilldown: function(aggregateKey) {
        return this.filter(function(notification) {
            return (notification.get("aggregateKey") == aggregateKey);
        });
    },

    focused: function(focusPoint) {
        return this.find(function(notification) {
            return (notification.get(focusPoint));
        });
    },

    focusMain: function(direction, showAll) {
        var mainCollection = this.main(showAll),
            triggerName = "removeMainFocused",
            focusPoint = "mainFocused";
        return ("up" == direction) ? this.focusPrevious(mainCollection, focusPoint, triggerName) : this.focusNext(mainCollection, focusPoint, triggerName);
    },

    setFocused: function(notificationGroup) {
        this.globalEvents.trigger("removeMainFocused");
        notificationGroup.set("mainFocused", true);
    },

    focusNext: function(collection, focusPoint, triggerName) {
        var currIndex = _.indexOf(collection, this.focused(focusPoint));
        if (currIndex < (collection.length - 1)) {
            if (triggerName) {
                this.globalEvents.trigger(triggerName);
            } else {
                currentView[currIndex].set(focusPoint, false);
            }
            collection[currIndex + 1].set(focusPoint, true);
            return true;
        } else {
            return false;
        }
    },

    focusPrevious: function(currentView, focusPoint, triggerName) {
        var currIndex = _.indexOf(currentView, this.focused(focusPoint));
        if (currIndex > 0) {
            if (triggerName) {
                this.globalEvents.trigger(triggerName);
            } else {
                currentView[currIndex].set(focusPoint, false);
            }
            currentView[currIndex - 1].set(focusPoint, true);
            return true;
        }
        return false;
    },

    countNested: function(items) {
        var count = 0;
        _(items).each(function(item) {
            count += item.get('notifications').length;
        });
        return count;
    }
});
