package com.atlassian.multitenant.hibernate2;

import com.atlassian.multitenant.MultiTenantContext;
import com.atlassian.multitenant.cache.MultiTenantCacheKey;
import net.sf.hibernate.cache.Cache;
import net.sf.hibernate.cache.CacheException;
import net.sf.hibernate.cache.CacheProvider;
import net.sf.hibernate.cache.EhCacheProvider;

import java.util.Properties;

/**
 * Shared Eh Cache Provider that can safely be used in a multi tenant context.  This implementation is inefficient
 * because clearing will clear for all tenants.  Additionally, lock contention between the tenants may be a problem.
 */
public class SharedMultiTenantEhCacheProvider implements CacheProvider
{
    private final CacheProvider delegate;

    public SharedMultiTenantEhCacheProvider()
    {
        this.delegate = new EhCacheProvider();
    }

    public Cache buildCache(final String regionName, final Properties properties) throws CacheException
    {
        return new SharedMultiTenantEhCache(delegate.buildCache(regionName, properties));
    }

    public long nextTimestamp()
    {
        return delegate.nextTimestamp();
    }

    public void start(final Properties properties) throws CacheException
    {
        delegate.start(properties);
    }

    public void stop()
    {
        delegate.stop();
    }

    private class SharedMultiTenantEhCache implements Cache
    {
        private final Cache cache;

        private SharedMultiTenantEhCache(final Cache cache)
        {
            this.cache = cache;
        }

        public Object get(final Object key) throws CacheException
        {
            return cache.get(createKey(key));
        }

        public void put(final Object key, final Object value) throws CacheException
        {
            cache.put(createKey(key), value);
        }

        public void remove(final Object key) throws CacheException
        {
            cache.remove(createKey(key));
        }

        public void clear() throws CacheException
        {
            cache.clear();
        }

        public void destroy() throws CacheException
        {
            cache.destroy();
        }

        public void lock(final Object key) throws CacheException
        {
            cache.lock(createKey(key));
        }

        public void unlock(final Object key) throws CacheException
        {
            cache.unlock(createKey(key));
        }

        public long nextTimestamp()
        {
            return cache.nextTimestamp();
        }

        public int getTimeout()
        {
            return cache.getTimeout();
        }

        private MultiTenantCacheKey createKey(Object key)
        {
            return new MultiTenantCacheKey(MultiTenantContext.getTenantReference().get().getName(), key);
        }
    }

}
