package com.atlassian.multitenant.spring;

import com.atlassian.multitenant.MultiTenantContext;
import org.springframework.beans.MutablePropertyValues;
import org.springframework.beans.factory.config.BeanDefinition;
import org.springframework.beans.factory.config.BeanDefinitionHolder;
import org.springframework.beans.factory.support.AbstractBeanDefinition;
import org.springframework.beans.factory.support.BeanDefinitionReaderUtils;
import org.springframework.beans.factory.support.BeanDefinitionRegistry;
import org.springframework.beans.factory.support.ChildBeanDefinition;
import org.springframework.beans.factory.support.RootBeanDefinition;
import org.springframework.beans.factory.xml.BeanDefinitionDecorator;
import org.springframework.beans.factory.xml.ParserContext;
import org.w3c.dom.Attr;
import org.w3c.dom.Element;
import org.w3c.dom.Node;

/**
 * Decorator for multitenant stateful beans
 */
public class MultiTenantStatefulBeanDefinitionDecorator implements BeanDefinitionDecorator
{
    private static final String TARGET_BEAN_SUFFIX = ".multitenant.target";

    public BeanDefinitionHolder decorate(final Node node, final BeanDefinitionHolder beanDefinitionHolder,
            final ParserContext parserContext)
    {
        final String stateful = ((Attr) node).getValue();
        // Only do something if multitenancy is enabled, and the attribute is set to true
        if (!(Boolean.parseBoolean(stateful) && MultiTenantContext.isEnabled()))
        {
            return beanDefinitionHolder;
        }
        BeanDefinitionRegistry registry = parserContext.getRegistry();

        String beanName = beanDefinitionHolder.getBeanName();
        String[] aliases = beanDefinitionHolder.getAliases();
        BeanDefinition existingDefinition = beanDefinitionHolder.getBeanDefinition();
        String className = existingDefinition.getBeanClassName();

        if (!(existingDefinition instanceof AbstractBeanDefinition))
        {
            throw new IllegalArgumentException("Cannot make a non AbstractBeanDefinition stateful");
        }

        String newBeanName = beanName + TARGET_BEAN_SUFFIX;

        AbstractBeanDefinition abstractDefinition = (AbstractBeanDefinition) existingDefinition;

        // Override a few things on the original one
        // Remove all the attributes, so that the new one doesn't get decorated by anything (including us!)
        for (String attribute : abstractDefinition.attributeNames())
        {
            abstractDefinition.removeAttribute(attribute);
        }
        // Should be prototype, so a new one is created each time (one for each tenant)
        abstractDefinition.setScope(RootBeanDefinition.SCOPE_PROTOTYPE);
        // We don't want this bean to be a candidate for autowiring
        abstractDefinition.setAutowireCandidate(false);

        // Register the new one under the multitenant target name
        BeanDefinitionReaderUtils.registerBeanDefinition(new BeanDefinitionHolder(abstractDefinition, newBeanName),
                registry);

        RootBeanDefinition factoryDefinition = new RootBeanDefinition();
        factoryDefinition.setBeanClass(MultiTenantFactoryBean.class);
        MutablePropertyValues mpvs = new MutablePropertyValues();
        factoryDefinition.setPropertyValues(mpvs);
        mpvs.addPropertyValue("targetName", newBeanName);
        mpvs.addPropertyValue("implementation", className);

        // Find out if it should be lazy loaded
        Element parent = ((Attr) node).getOwnerElement();
        Attr attr = parent.getAttributeNodeNS(MultiTenantNamespaceHandler.NAMESPACE, "lazy-loaded");
        if (attr != null && !Boolean.parseBoolean(attr.getValue()))
        {
            mpvs.addPropertyValue("lazyLoad", "false");
        }

        return new BeanDefinitionHolder(factoryDefinition, beanName, aliases);
    }
}
