package com.atlassian.multitenant.log4j;

import com.atlassian.multitenant.MultiTenantComponentMap;
import com.atlassian.multitenant.MultiTenantCreator;
import com.atlassian.multitenant.MultiTenantDestroyer;
import com.atlassian.multitenant.MultiTenantContext;
import com.atlassian.multitenant.Tenant;
import org.apache.log4j.ConsoleAppender;
import org.apache.log4j.Layout;
import org.apache.log4j.Logger;
import org.apache.log4j.RollingFileAppender;
import org.apache.log4j.WriterAppender;
import org.apache.log4j.spi.LoggingEvent;

import java.io.File;

/**
 * This appender will log to the home directory of the product, in a folder called logs.  The default file name if not
 * overridden is atlassian.log.  The appender
 */
public class MultiTenantHomeAppender extends WriterAppender
{
    private static final Logger log = Logger.getLogger(MultiTenantHomeAppender.class);

    /**
     * The default name of the log file
     */
    private final static String DEFAULT_LOG_NAME = "atlassian.log";
    private final static String LOGS_DIR = "logs";

    // All properties set must be cached for new appender instance creation
    private volatile Layout layout;
    private volatile String maxFileSize;
    private volatile int maxBackupIndex;
    private volatile String logFileName;
    private volatile boolean activated;

    private volatile MultiTenantComponentMap<RollingFileAppender> map;
    private final ConsoleAppender consoleAppender;

    public MultiTenantHomeAppender()
    {
        consoleAppender = new ConsoleAppender();
    }

    @Override
    public Layout getLayout()
    {
        return layout;
    }

    @Override
    public void setLayout(final Layout layout)
    {
        consoleAppender.setLayout(layout);
        this.layout = layout;
    }

    @Override
    public void close()
    {
        consoleAppender.close();
        activated = false;
        for (RollingFileAppender appender : map.getAll())
        {
            appender.close();
        }
    }

    @Override
    public void append(final LoggingEvent event)
    {
        // We don't want errors thrown from the logging system, so we are extra cautious here.  Firstly, if the map is
        // null, it means the application hasn't called the initialise method, so we log straight to the console.
        // If there is no multi tenant context, log straight to the console.  If the component is not
        // initialised, we are probably going through the start phase of a tenant, or the tenant has been stopped but
        // some thread still is executing in its context, so log to the console.
        if (map != null && MultiTenantContext.getTenantReference().isSet() && map.isInitialised())
        {
            RollingFileAppender appender = map.get();
            // The appender could be null, if there was a problem initialising it, log to the console if so.
            if (appender == null)
            {
                consoleAppender.append(event);
            }
            else
            {
                map.get().append(event);
            }
        }
        else
        {
            consoleAppender.append(event);
        }
    }

    public synchronized void initialise()
    {
        if (map != null)
        {
            throw new IllegalStateException("Initialised called twice");
        }
        map = MultiTenantContext.getFactory().createComponentMapBuilder(new RollingFileAppenderCreator())
                .setLazyLoad(MultiTenantComponentMap.LazyLoadStrategy.EAGER_LOAD).construct();
    }

    @Override
    public void activateOptions()
    {
        consoleAppender.activateOptions();
        activated = true;
        if (map != null)
        {
            // Very unlikely that we'll get here with map being non null, but make sure that if we happen to have
            // initialised it, that all the existing tenants get initialised.
            map.initialiseAll();
        }
    }

    public String getMaxFileSize()
    {
        return maxFileSize;
    }

    public void setMaxFileSize(final String maxFileSize)
    {
        this.maxFileSize = maxFileSize;
    }

    public int getMaxBackupIndex()
    {
        return maxBackupIndex;
    }

    public void setMaxBackupIndex(final int maxBackupIndex)
    {
        this.maxBackupIndex = maxBackupIndex;
    }

    public String getLogFileName()
    {
        return logFileName;
    }

    public void setLogFileName(final String logFileName)
    {
        this.logFileName = logFileName;
    }

    private class RollingFileAppenderCreator implements MultiTenantCreator<RollingFileAppender>,
            MultiTenantDestroyer<RollingFileAppender>
    {
        public RollingFileAppender create(final Tenant tenant)
        {
            // Activited will only be false if the logger has been closed, in which case, don't try and create a new one.
            if (activated)
            {
                String logsLocation = tenant.getHomeDir() + File.separator + LOGS_DIR;

                File logDirectory = new File(logsLocation);
                // log4j does not create the directory automatically
                if (!logDirectory.exists() && !logDirectory.mkdir())
                {
                    log.error("Could not create logs directory " + logsLocation +
                            ". Logging remains directed to the ConsoleAppender.");
                    return null;
                }

                if (logFileName == null)
                {
                    logFileName = DEFAULT_LOG_NAME;
                }

                String logFile = logsLocation + File.separator + logFileName;

                RollingFileAppender appender = new RollingFileAppender();
                if (maxBackupIndex != 0)
                {
                    appender.setMaxBackupIndex(maxBackupIndex);
                }
                if (maxFileSize != null)
                {
                    appender.setMaxFileSize(maxFileSize);
                }
                appender.setFile(logFile);
                appender.activateOptions();
                return appender;
            }
            return null;
        }


        public void destroy(final Tenant tenant, final RollingFileAppender instance)
        {
            instance.close();
        }
    }
}