package com.atlassian.multitenant.hibernate2;

import com.atlassian.multitenant.MultiTenantCreator;
import com.atlassian.multitenant.MultiTenantDestroyer;
import com.atlassian.multitenant.MultiTenantComponentMap;
import com.atlassian.multitenant.MultiTenantContext;
import com.atlassian.multitenant.Tenant;
import com.atlassian.multitenant.hibernate.HibernateMultiTenantConfig;
import net.sf.hibernate.HibernateException;
import net.sf.hibernate.cfg.Environment;
import net.sf.hibernate.connection.ConnectionProvider;
import net.sf.hibernate.connection.ConnectionProviderFactory;
import org.apache.log4j.Logger;

import java.sql.Connection;
import java.sql.SQLException;
import java.util.Properties;

/**
 * This is a Hibernate connection provider that delegates to the correct connection for the tenant.  It will
 * automatically create new connection providers when a tenant is added.
 * <p/>
 * If created while a multi tenant context is set, then the provider will assume that it is an ad hoc provider being
 * created only for a single tenant (as may be the case when doing a schema export).  In which case, it will act purely
 * as a delegate to an actual provider for that tenant, no map of tenants to providers will be created.
 * <p/>
 * The provider caches the properties that are initially sent to it from Hibernate, and uses those as a basis for
 * configuring all the new providers.  For example, c3p0 properties that are configured initially are used here.
 * <p/>
 * To use this connection provider, simply set hibernate.connection.provider_class to the name of this class.
 */
public class MultiTenantConnectionProvider implements ConnectionProvider
{
    private static final Logger log = Logger.getLogger(MultiTenantConnectionProvider.class);
    /**
     * This is the map of tenants to connection providers, if set to null, then the connection provider is in
     * single tenant mode, and uses the delegate field instead.
     */
    private final MultiTenantComponentMap<ConnectionProvider> map;
    private volatile Properties initialProperties;
    private volatile ConnectionProvider delegate;

    public MultiTenantConnectionProvider()
    {
        if (MultiTenantContext.getTenantReference().isSet())
        {
            // Assume we're an ad hoc provider for a single tenant, set the map to null
            map = null;
        }
        else
        {
            ConnectionProviderCreator creator = new ConnectionProviderCreator();
            // The map must be eager after init, so that new connection providers aren't created until configure has
            // been called with a set of properties to configure with.  It is assumed that configure will be called
            // immediately after this constructor has returned.  See ConnectionProviderFactory.newConnectionProvider(),
            // that should be the only code that constructs this provider.
            map = MultiTenantContext.getFactory().createComponentMapBuilder(creator).setDestroyer(creator)
                    .setLazyLoad(MultiTenantComponentMap.LazyLoadStrategy.EAGER_AFTER_STARTUP).construct();
        }
    }

    public void configure(final Properties props) throws HibernateException
    {
        this.initialProperties = props;
        if (map == null)
        {
            // Instantiate the delegate
            delegate = new ConnectionProviderCreator().create(MultiTenantContext.getTenantReference().get());
        }
        else
        {
            map.initialiseAll();
        }
    }

    public Connection getConnection() throws SQLException
    {
        if (map == null)
        {
            return delegate.getConnection();
        }
        //if (MultiTenantContext.getTenantReference().isInitialising())
        //{
        //    return new ConnectionPlaceholder();
        //}
        return map.get().getConnection();
    }

    public void closeConnection(final Connection conn) throws SQLException
    {
        if (map == null)
        {
            delegate.closeConnection(conn);
        }
        //else if (!MultiTenantContext.getTenantReference().isInitialising())
        {
            map.get().closeConnection(conn);
        }
    }

    public void close() throws HibernateException
    {
        if (map == null)
        {
            delegate.close();
        }
        else
        {
            for (ConnectionProvider provider : map.getAll())
            {
                provider.close();
            }
        }
    }

    private class ConnectionProviderCreator implements MultiTenantCreator<ConnectionProvider>,
            MultiTenantDestroyer<ConnectionProvider>
    {
        public ConnectionProvider create(final Tenant tenant)
        {
            Properties properties = new Properties();
            for (String property : initialProperties.stringPropertyNames())
            {
                properties.setProperty(property, initialProperties.getProperty(property));
            }

            // Remove the connection provider specification, and make sure there's no datasource set
            properties.remove(Environment.CONNECTION_PROVIDER);
            properties.remove(Environment.DATASOURCE);

            HibernateMultiTenantConfig config = tenant.getConfig(HibernateMultiTenantConfig.class);
            properties.setProperty(Environment.DRIVER, config.getDriverClass().getName());
            properties.setProperty(Environment.URL, config.getDatabaseUrl());
            properties.setProperty(Environment.USER, config.getUsername());
            properties.setProperty(Environment.PASS, config.getPassword());
            properties.setProperty(Environment.C3P0_MAX_SIZE, Integer.toString(config.getPoolSize()));

            try
            {
                return ConnectionProviderFactory.newConnectionProvider(properties);
            }
            catch (HibernateException e)
            {
                throw new RuntimeException("Error starting Hibernate provider", e);
            }
        }

        public void destroy(final Tenant tenant, final ConnectionProvider connectionProvider)
        {
            try
            {
                connectionProvider.close();
            }
            catch (HibernateException he)
            {
                log.warn("Error shutting down connection provider for tenant: " + tenant.getName(), he);
            }
        }
    }
}
