package com.atlassian.multitenant.cache;

import com.atlassian.cache.Cache;
import com.atlassian.cache.CacheManager;
import com.atlassian.multitenant.MultiTenantContext;
import com.atlassian.util.concurrent.CopyOnWriteMap;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Map;
import java.util.Set;

/**
 * Single cache manager that can safely be shared between all tenants
 */
public class SharedMultiTenantCacheManager implements CacheManager
{
    private final CacheManager delegate;
    private final Map<String, Cache> caches = CopyOnWriteMap.newHashMap();
    private final Set<Object> sharedCaches;

    public SharedMultiTenantCacheManager(final CacheManager delegate, final Set<Object> sharedCaches)
    {
        this.delegate = delegate;
        this.sharedCaches = sharedCaches;
    }

    public Collection getCaches()
    {
        return delegate.getCaches();
    }

    public void flushCaches()
    {
        delegate.flushCaches();
    }

    public Cache getCache(final String name)
    {
        Cache cache = caches.get(name);
        if (cache == null)
        {
            synchronized (this)
            {
                if (!caches.containsKey(name))
                {
                    if (sharedCaches.contains(name))
                    {
                        cache = delegate.getCache(name);
                    }
                    else
                    {
                        cache = new SharedMultiTenantCache(name);
                    }
                    caches.put(name, cache);
                }
            }
        }
        return cache;
    }

    private class SharedMultiTenantCache implements Cache
    {
        private final String name;
        private final Cache cache;

        private SharedMultiTenantCache(final String name)
        {
            this.name = name;
            this.cache = delegate.getCache(name);
        }

        public String getName()
        {
            return name;
        }

        public Collection getKeys()
        {
            Collection results = new ArrayList();
            String tenant = MultiTenantContext.getTenantReference().get().getName();
            for (MultiTenantCacheKey key : ((Iterable<MultiTenantCacheKey>) cache.getKeys()))
            {
                if (key.getTenant().equals(tenant))
                {
                    results.add(key);
                }
            }
            return results;
        }

        public Object get(final Object key)
        {
            return cache.get(createKey(key));
        }

        public void put(final Object key, final Object value)
        {
            cache.put(createKey(key), value);
        }

        public void remove(final Object key)
        {
            cache.remove(createKey(key));
        }

        public void removeAll()
        {
            String tenant = MultiTenantContext.getTenantReference().get().getName();
            for (MultiTenantCacheKey key : ((Iterable<MultiTenantCacheKey>) cache.getKeys()))
            {
                if (key.getTenant().equals(tenant))
                {
                    cache.remove(key);
                }
            }
        }

        private MultiTenantCacheKey createKey(Object key)
        {
            return new MultiTenantCacheKey(MultiTenantContext.getTenantReference().get().getName(), key);
        }
    }

}
