package com.atlassian.multitenant.cache;

import com.atlassian.cache.Cache;
import com.atlassian.cache.CacheManager;
import com.atlassian.multitenant.MultiTenantContext;
import com.atlassian.util.concurrent.CopyOnWriteMap;

import java.util.Collection;
import java.util.Collections;
import java.util.Map;
import java.util.Set;

/**
 * This cache manager returns its own list of caches that delegate to the underlying cache manager.  The underlying
 * cache manager must be multi tenant aware, this cache manager only serves to defer cache creation until the cache is
 * actually used, rather than when the getCache() method is used.  That way getCache() can be called in a non multi
 * tenant context, eg during initialisation.
 */
public class MultiTenantCacheManager implements CacheManager
{
    private final CacheManager cacheManager;
    private final Map<String, Cache> caches = CopyOnWriteMap.newHashMap();
    private final Set<Object> sharedCaches;

    public MultiTenantCacheManager(final CacheManager cacheManager, final Set<Object> sharedCaches)
    {
        this.cacheManager = cacheManager;
        this.sharedCaches = sharedCaches;
    }

    public Collection getCaches()
    {
        return caches.values();
    }

    public void flushCaches()
    {
        cacheManager.flushCaches();
    }

    public Cache getCache(final String name)
    {
        Cache cache = caches.get(name);
        if (cache == null)
        {
            synchronized (this)
            {
                if (!caches.containsKey(name))
                {
                    if (sharedCaches.contains(name))
                    {
                        cache = cacheManager.getCache(name);
                    }
                    else
                    {
                        cache = new MultiTenantCache(name);
                    }
                    caches.put(name, cache);
                }
            }
        }
        return cache;
    }

    /**
     * A multi tenant cache implementation.  If no context is set, the cache is disabled.
     */
    private class MultiTenantCache implements Cache
    {
        private final String name;

        private MultiTenantCache(final String name)
        {
            this.name = name;
        }

        public String getName()
        {
            return name;
        }

        private Cache getCache()
        {
            return cacheManager.getCache(name);
        }

        public Collection getKeys()
        {
            if (MultiTenantContext.getTenantReference().isSet())
            {
                return getCache().getKeys();
            }
            else
            {
                return Collections.emptySet();
            }
        }

        public Object get(final Object key)
        {
            if (MultiTenantContext.getTenantReference().isSet())
            {
                return getCache().get(key);
            }
            else
            {
                return null;
            }
        }

        public void put(final Object key, final Object value)
        {
            if (MultiTenantContext.getTenantReference().isSet())
            {
                getCache().put(key, value);
            }
        }


        public void remove(final Object key)
        {
            if (MultiTenantContext.getTenantReference().isSet())
            {
                getCache().remove(key);
            }
        }

        public void removeAll()
        {
            if (MultiTenantContext.getTenantReference().isSet())
            {
                getCache().removeAll();
            }
        }


    }
}
