package com.atlassian.messagequeue;

import com.atlassian.annotations.PublicApi;

import java.util.Optional;

import static java.util.Objects.requireNonNull;

/**
 * Encapsulates information that is passed to a message invocation.
 * @since 1.0
 */
@PublicApi
public final class Message {

    private final MessageRunnerKey runnerKey;
    private final Optional<String> payload;

    private Message(MessageRunnerKey runnerKey, Optional<String> payload) {
        this.runnerKey = requireNonNull(runnerKey, "runnerKey");
        this.payload = requireNonNull(payload, "payload");
    }

    /**
     * @return message runner key
     */
    public MessageRunnerKey getRunnerKey() {
        return runnerKey;
    }

    /**
     * @return message payload
     */
    public Optional<String> getPayload() {
        return payload;
    }

    /**
     * Message factory method
     * @param runnerKey message runner key
     * @param payload message payload
     * @return Message instance
     */
    public static Message create(MessageRunnerKey runnerKey, String payload) {
        return builder(runnerKey).payload(payload).build();
    }

    /**
     * Builder factory method
     * @param runnerKey message runner key
     * @return Message Builder
     */
    public static Builder builder(MessageRunnerKey runnerKey) {
        return new Builder(runnerKey);
    }

    /**
     * Message Builder
     */
    @PublicApi
    public static class Builder {
        private final MessageRunnerKey runnerKey;
        private Optional<String> payload = Optional.empty();

        /**
         * @param runnerKey message runner key
         */
        public Builder(MessageRunnerKey runnerKey) {
            this.runnerKey = requireNonNull(runnerKey, "runnerKey");
        }

        /**
         * @return a message instance
         */
        public Message build() {
            return new Message(runnerKey, payload);
        }

        /**
         * @param payload message payload
         * @return message
         */
        public Builder payload(String payload) {
            this.payload = Optional.ofNullable(payload);
            return this;
        }
    }
}