package com.atlassian.maven.plugins;

import java.io.File;
import java.io.FileFilter;
import java.util.HashSet;
import java.util.Set;

import org.apache.commons.io.filefilter.WildcardFileFilter;
import org.apache.maven.plugin.MojoExecutionException;

public class PathParserUtil
{

	/**
	 * Takes a list of paths and converts them to a set of File objects.
	 * Path can be either a relative or absolute path to a file or directory or a reference to children of a directory using '*' or '?'. For instance:
	 * <ul>
	 * <li> dir1/dir2/file.txt </li>
	 * <li> dir1/dir2/ </li>
	 * <li> dir1/dir2 </li>
	 * <li> dir1/dir2/*.txt </li>
	 * <li> dir1/dir2/Us*Nam?.* </li>
	 * </ul>
	 * 
	 * The wildcard '*' must not occur within the path but only at the end of the path. I.e. the following paths are unsupported:
	 * <ul>
	 * <li> dir1/*\/dir2 </li>
	 * <li> *\/dir2 </li>
	 * </ul>
	 * 
	 * @param paths
	 * @return
	 * @throws MojoExecutionException
	 */
	static Set toFiles(String[] paths) throws MojoExecutionException
	{
		Set resources = new HashSet();
		for (int i = 0; i < paths.length; i++)
		{
			Set nextSet = toFiles(paths[i]);
			if (nextSet != null) resources.addAll(nextSet);

		}
		return resources;
	}

	/**
	 * Parses the given file path into one or more File objects.
	 * Path can be either a relative or absolute path to a file or directory or a reference to children of a directory using '*' or '?'. For instance:
	 * <ul>
	 * <li> dir1/dir2/file.txt </li>
	 * <li> dir1/dir2/ </li>
	 * <li> dir1/dir2 </li>
	 * <li> dir1/dir2/*.txt </li>
	 * <li> dir1/dir2/Us*Nam?.* </li>
	 * </ul>
	 * 
	 * The wildcard '*' must not occur within the path but only at the end of the path. I.e. the following paths are unsupported:
	 * <ul>
	 * <li> dir1/*\/dir2 </li>
	 * <li> dir1/*\/dir2 </li>
	 * <li> dir1/Us*Name?/* </li>
	 * <li> *\/dir?/file.txt </li>
	 * </ul>
	 * 
	 * @param pathStr
	 * @return
	 * @throws MojoExecutionException
	 */
	static Set toFiles(final String pathStr) throws MojoExecutionException
	{
		if (pathStr.length() == 0) return null;

		ResourceDescriptor descr = new ResourceDescriptor(pathStr);
		Set resources = new HashSet();

		File resource = new File(descr.path);
		if (resource.exists())
        {
            if (descr.wildcard != null)
            {
                if (resource.isDirectory())
                {
                    getMatchingChildren(resource, descr.wildcard, resources);
                }
            }
            else
            {
                resources.add(resource);
            }
        }

        return resources;
	}

	static void getMatchingChildren(File resource, String wildcard, Set resources)
	{
		FileFilter filter = new WildcardFileFilter(wildcard);
		File children[] = resource.listFiles(filter);

		for (int i = 0; i < children.length; i++)
			resources.add(children[i]);
	}

	/**
	 * Looks for the first occurrence of either '?' or '*' character and returns its position. Otherwise returns -1;
	 * @param path
	 * @return position of the first occurrence of either '?' or '*' character or -1;
	 */
	static int findFirstGlobCharPosition(final String path)
	{
		char[] array = path.toCharArray();
		for (int i = 0; i < array.length; i++)
		{
			if (array[i] == '?' || array[i] == '*') return i;
		}
		return -1;
	}

	/**
	 * Represents a path to one or more resources.
	 */
	static class ResourceDescriptor
	{
		final String path;
		final String wildcard;

		/**
		 * Parses the given file path into one or more File objects.
		 * Path can be either a relative or absolute path to a file or directory or a reference to children of a directory using '*' or '?'. For instance:
		 * <ul>
		 * <li> dir1/dir2/file.txt </li>
		 * <li> dir1/dir2/ </li>
		 * <li> dir1/dir2 </li>
		 * <li> dir1/dir2/*.txt </li>
		 * <li> dir1/dir2/Us*Nam?.* </li>
	 	 * </ul>
	 	 *  
		 * The wildcard '*' must not occur within the path but only at the end of the path. I.e. the following paths are unsupported:
		 * <ul>
		 * <li> dir1/*\/dir2 </li>
		 * <li> dir1/*\/dir2 </li>
		 * <li> dir1/Us*Name?/* </li>
		 * <li> *\/dir?/file.txt </li>
		 * </ul>
		 * 
		 * @throws MojoExecutionException
		 */
		public ResourceDescriptor(String path) throws MojoExecutionException
		{
			int pos = findFirstGlobCharPosition(path);

			if (pos != -1)
			{
				int dirEndPos = path.replace('\\', '/').lastIndexOf('/');
				if (dirEndPos >= pos) throw new MojoExecutionException("Invalid path - '" + path
				        + "'. Wildcards must not contain a path separator '/' or '\\'.");
				else if (dirEndPos == -1)
				{
					// children of the current dir are being matched
					this.path = "";
					this.wildcard = path;
				}
				else
				{
					this.path = path.substring(0, dirEndPos + 1);
					this.wildcard = path.substring(dirEndPos + 1);
				}
			}
			else
			{
				this.path = path;
				this.wildcard = null;
			}
		}
	}
}