package com.atlassian.maven.plugins.amps;

import java.io.File;
import java.util.Map;

import org.apache.maven.plugin.AbstractMojo;
import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugin.MojoFailureException;
import org.apache.maven.plugin.logging.Log;
import org.apache.maven.plugins.annotations.Mojo;
import org.apache.maven.plugins.annotations.Parameter;
import com.google.common.annotations.VisibleForTesting;

import com.atlassian.upm.signing.tool.UpmConfigHandler;

@Mojo(name = "create-upm-config")
public class CreateUpmConfigMojo extends AbstractMojo {

    public static final String ATLASSIAN_UPM_CONFIGURATION_FILE = "atlassian.upm.configuration.file";
    public static final String ATLASSIAN_UPM_TRUSTSTORE_DIRECTORY = "atlassian.upm.truststore.directory";
    public static final String ATLASSIAN_UPM_CONFIGURATION_DIRECTORY = "atlassian.upm.configuration.directory";
    public static final String ATLASSIAN_UPM_CONFIG_LOOSECHECK_ALLOWED = "atlassian.upm.config.loosecheck.allowed";

    /** The project's build directory. */
    @Parameter(property = "project.build.directory", required = true)
    protected File targetDirectory;

    /** Enables UPM's plugin signature check feature in the product */
    @Parameter(property = "enablePluginSigning", required = true, defaultValue = "false")
    @VisibleForTesting
    private boolean enablePluginSigning;

    /** if true, UPM configuration will be made readonly when created. True by default, as required by upm &lt; 8. */
    @Parameter(property = "upmConfigReadonly", required = true, defaultValue = "true")
    @VisibleForTesting
    protected boolean upmConfigReadonly;
    /**
     * UPM's plugin signature check feature configuration path. If omitted, ${project.target.directory}/upmconfig will
     * be used.
     */
    @Parameter(property = "upmConfigPath")
    @VisibleForTesting
    private String upmConfigPath;

    @Parameter(property = "upmConfigFile")
    @VisibleForTesting
    private String upmConfigFile;

    @Parameter(property = "upmTrustStore")
    @VisibleForTesting
    private String upmTrustStore;
    /** If true, AMPS will overwrite an existing UPM plugin signature check configuration */
    @Parameter(property = "overwriteUpmConfig", required = true, defaultValue = "false")
    @VisibleForTesting
    private boolean overwriteUpmConfig;

    /**
     * Qualifier of Atlassians certificates to install in the truststore. Can be STAGING (staging Marketplace
     * certificates), PROD (production marketplace certificates), or ALL (all available certificates)
     */
    @Parameter(property = "atlassianCertificates")
    @VisibleForTesting
    private String atlassianCertificates;

    /**
     * Useful to add a custom certificate and private key, which can be later used to sign plugins. The following
     * Properties must be provided :
     *
     * <ul>
     *   <li>issuer (required) : the certificate issuer
     *   <li>name (optional, defaults to issuer with spaces replaced with '_') : the symbolic name of the
     *       certificate/key pair. Used as file name prefix, and to load the pair from test code in order to sign
     *       plugins.
     *   <li>startOffset (optional, defaults to 0) : a duration (ISO-8601 notation), with an optional trailing '-' .
     *       This duration is added (or removed if the "-' sign is present) from the current date and used to set the
     *       certificate's validity start date.
     *   <li>duration (optional, defaults to 1 day) : duration of the certificate
     * </ul>
     */
    @Parameter(property = "customCertificate")
    private CustomCertificateBuilder customCertificate;

    @Override
    public void execute() throws MojoExecutionException, MojoFailureException {
        UpmConfigHandler handler = UpmConfigHandler.builder(targetDirectory.toPath())
                .enabled(enablePluginSigning)
                .readonly(upmConfigReadonly)
                .upmConfigPath(upmConfigPath)
                .upmConfigFilePath(upmConfigFile)
                .upmTrustStorePath(upmTrustStore)
                .overwriteUpmConfig(overwriteUpmConfig)
                .atlassianCertificates(atlassianCertificates)
                .customCertificate(customCertificate)
                .build();
        handler.writeConfiguration();
        Map<String, String> properties = handler.getSystemProperties();

        Log log = getLog();
        String singleRoot = properties.get(ATLASSIAN_UPM_CONFIGURATION_DIRECTORY);
        String configFile = properties.get(ATLASSIAN_UPM_CONFIGURATION_FILE);
        String truststore = properties.get(ATLASSIAN_UPM_TRUSTSTORE_DIRECTORY);
        if (singleRoot != null) {
            log.info("UPM configuration was written to " + singleRoot);
        } else {
            log.info("UPM configuration file was written to " + configFile);
            log.info("UPM truststore was created in directory " + truststore);
        }
        StringBuilder message = new StringBuilder(
                "Add the following system properties on product startup to use this configuration : ");
        properties.forEach(
                (key, value) -> message.append("\n\t").append(key).append("=").append(value));
        log.info(message.toString());
    }
}
