package com.atlassian.maven.plugins.amps;

import com.atlassian.maven.plugins.amps.product.AmpsDefaults;
import com.atlassian.maven.plugins.amps.product.ProductHandler;
import com.google.common.annotations.VisibleForTesting;
import org.apache.commons.lang3.StringUtils;
import org.apache.maven.artifact.handler.manager.ArtifactHandlerManager;
import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugins.annotations.Component;
import org.apache.maven.plugins.annotations.Mojo;
import org.apache.maven.plugins.annotations.Parameter;
import org.apache.maven.plugins.annotations.ResolutionScope;
import org.apache.maven.project.MavenProject;

import java.io.File;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;

import static com.atlassian.maven.plugins.amps.MavenGoals.getReportsDirectory;
import static java.util.Collections.emptyList;
import static java.util.Collections.emptyMap;
import static java.util.Collections.singleton;
import static java.util.Collections.singletonList;
import static org.apache.commons.lang3.StringUtils.isBlank;
import static org.apache.commons.lang3.StringUtils.trimToEmpty;

/**
 * Run the integration tests against the webapp.
 */
@Mojo(name = "integration-test", requiresDependencyResolution = ResolutionScope.TEST)
public class IntegrationTestMojo extends AbstractTestGroupsHandlerMojo {
    /**
     * Pattern for to use to find integration tests. Only used if no test groups are defined.
     */
    @Parameter(property = "functional.test.pattern")
    private String functionalTestPattern = MavenGoals.REGEX_INTEGRATION_TESTS;

    /**
     * The directory containing generated test classes of the project being tested.
     */
    @Parameter(property = "project.build.testOutputDirectory", required = true)
    private File testClassesDirectory;

    /**
     * A comma separated list of test groups to run. If not specified, all test groups are run.
     */
    @Parameter(property = "testGroups")
    private String configuredTestGroupsToRun;

    /**
     * If true, no products will be started.
     */
    @Parameter(property = "no.webapp", defaultValue = "false")
    private boolean noWebapp;

    @Component
    private ArtifactHandlerManager artifactHandlerManager;

    /**
     * Whether to skip the integration tests along with any product startups (one of three synonymous flags).
     */
    @Parameter(property = "maven.test.skip", defaultValue = "false")
    private boolean testsSkip;

    /**
     * Whether to skip the integration tests along with any product startups (one of three synonymous flags).
     */
    @Parameter(property = "skipTests", defaultValue = "false")
    private boolean skipTests;

    /**
     * Whether to skip the integration tests along with any product startups (one of three synonymous flags).
     */
    @Parameter(property = "skipITs", defaultValue = "false")
    private boolean skipITs;

    /**
     * If any products are started, this is the debug port for the first product. Any other products this goal starts
     * will be given a debug port one higher than the previous product's debug port.
     */
    @Parameter(property = "jvm.debug.port", defaultValue = "0")
    protected int jvmDebugPort;

    /**
     * Whether to suspend the started products' JVMs until a remote debugger is attached.
     */
    @Parameter(property = "jvm.debug.suspend")
    protected boolean jvmDebugSuspend;

    /**
     * Passed as-is to the {@code failsafe:integration-test} goal as its {@code debugForkedProcess} parameter.
     */
    @Parameter(property = "maven.failsafe.debug")
    protected String mavenFailsafeDebug;

    /**
     * The test category as defined by the surefire/failsafe notion of groups. In JUnit4, this affects tests annotated
     * with the {@link org.junit.experimental.categories.Category Category} annotation.
     */
    @Parameter
    protected String category;

    /**
     * By default, this goal performs both failsafe:integration-test and failsafe:verify goals. Set this parameter to
     * {@code true} if you want to defer {@code failsafe:verify} to a later lifecycle phase (e.g. {@code verify}), in
     * order to perform cleanup in the {@code post-integration-test} phase. Please note that you will have to set the
     * execution(s) for failsafe:verify yourself in the pom.xml file, including the {@code reportsDirectory}
     * configuration for the {@code verify} goal.
     */
    @Parameter(property = "skip.IT.verification")
    protected boolean skipITVerification;

    protected void doExecute() throws MojoExecutionException {
        // Should this check for ITs respect the `functionalTestPattern` parameter?
        if (!new File(testClassesDirectory, "it").exists()) {
            getLog().info("No integration tests found");
            return;
        }

        if (skipTests || testsSkip || skipITs) {
            getLog().info("Integration tests skipped");
            return;
        }

        final MavenProject project = getMavenContext().getProject();

        // Workaround for MNG-1682/MNG-2426: force Maven to install artifacts using the "jar" handler
        project.getArtifact().setArtifactHandler(artifactHandlerManager.getArtifactHandler("jar"));

        final MavenGoals goals = getMavenGoals();
        final String pluginJar = targetDirectory.getAbsolutePath() + "/" + finalName + ".jar";

        for (String testGroupId : getTestGroupsToRun()) {
            runTestsForTestGroup(testGroupId, goals, pluginJar, copy(systemPropertyVariables));
        }
    }

    private Collection<String> getTestGroupsToRun() {
        final Collection<String> superclassTestGroupIds = getTestGroupIds();
        if (superclassTestGroupIds.isEmpty()) {
            return singleton(NO_TEST_GROUP);
        } else if (configuredTestGroupsToRun == null) {
            // No test groups configured for this goal, use those configured for the superclass
            return superclassTestGroupIds;
        } else {
            // Find the test groups configured for this goal that are valid according to the superclass
            final Collection<String> testGroupIdsInCommonWithSuperclass = new HashSet<>();
            for (String testGroupId : configuredTestGroupsToRun.split(",")) {
                if (superclassTestGroupIds.contains(testGroupId)) {
                    testGroupIdsInCommonWithSuperclass.add(testGroupId);
                } else {
                    getLog().warn("Test group '" + testGroupId + "' does not exist");
                }
            }
            return testGroupIdsInCommonWithSuperclass;
        }
    }

    private static Map<String, Object> copy(final Map<String, Object> mapIn) {
        return new HashMap<>(mapIn);
    }

    /**
     * Returns product-specific properties to pass to the container during
     * integration testing. Default implementation does nothing.
     *
     * @param product the {@code Product} object to use
     * @return a {@code Map} of properties to add to the system properties passed
     * to the container
     */
    protected Map<String, String> getProductFunctionalTestProperties(Product product) {
        return emptyMap();
    }

    private void runTestsForTestGroup(final String testGroupId, final MavenGoals goals, final String pluginJar,
                                      final Map<String, Object> systemProperties) throws MojoExecutionException {
        final List<String> includes = getIncludesForTestGroup(testGroupId);
        final List<String> excludes = getExcludesForTestGroup(testGroupId);

        final List<ProductExecution> productExecutions = getTestGroupProductExecutions(testGroupId);
        setParallelMode(productExecutions);

        final List<String> instanceIds = new ArrayList<>();

        final Map<Integer, ProductExecution> productExecutionWithActualPorts = new HashMap<>();
        int counter = 0;
        // Install the plugin in each product and start it
        for (final ProductExecution productExecution : productExecutions) {
            final ProductHandler productHandler = productExecution.getProductHandler();
            final Product product = productExecution.getProduct();
            instanceIds.add(product.getInstanceId());
            if (product.isInstallPlugin() == null) {
                product.setInstallPlugin(installPlugin);
            }

            if (shouldBuildTestPlugin()) {
                final List<ProductArtifact> plugins = product.getBundledArtifacts();
                plugins.addAll(getTestFrameworkPlugins());
            }

            final int actualHttpPort;
            if (noWebapp) {
                actualHttpPort = product.getHttpPort();
                if (actualHttpPort <= 0) {
                    throw new MojoExecutionException("Http server port must be set when using no.webapp flag.");
                }
            } else {
                if (jvmDebugPort > 0) {
                    counter = setUpDebugging(counter, product);
                }
                actualHttpPort = productHandler.start(product);
            }

            // when running with -Dno.webapp, then actualHttpPort may not reflect real conditions
            if (productExecutionWithActualPorts.put(actualHttpPort, productExecution) != null) {
                throw new MojoExecutionException("Http server port was already occupied");
            }
        }

        populateProperties(systemProperties, testGroupId, pluginJar, instanceIds, productExecutionWithActualPorts);

        if (!noWebapp) {
            waitForProducts(productExecutions, true);
        }

        MojoExecutionException thrown = null;
        try {
            // Actually run the tests
            final String reportsDirectory =
                    getReportsDirectory(targetDirectory, "group-" + testGroupId, getClassifier(testGroupId));
            goals.runIntegrationTests(
                    reportsDirectory, includes, excludes, systemProperties, category, mavenFailsafeDebug);
            if (skipITVerification) {
                getLog().info("Skipping failsafe IT failure verification.");
            } else {
                goals.runVerify(reportsDirectory);
            }
        } catch (MojoExecutionException e) {
            // If any tests fail an exception will be thrown. We need to catch that and hold onto it, because
            // even if tests fail any running products still need to be stopped
            thrown = e;
        } finally {
            if (!noWebapp) {
                try {
                    // Shut all products down.
                    stopProducts(productExecutions);
                } catch (MojoExecutionException e) {
                    if (thrown == null) {
                        // If no exception was thrown during the tests, propagate the failure to stop
                        thrown = e;
                    } else {
                        // Otherwise, suppress the stop failure and focus on the test failure
                        thrown.addSuppressed(e);
                    }
                }
            }
        }

        if (thrown != null) {
            // If tests failed, or if any products could not be stopped, propagate the exception
            throw thrown;
        }
    }

    @VisibleForTesting
    void populateProperties(
            final Map<String, Object> systemProperties, final String testGroupId, final String pluginJar,
            final List<String> instanceIds, final Map<Integer, ProductExecution> productExecutionsByWebPort) {
        putIfNotOverridden(systemProperties, "plugin.jar", pluginJar);
        putIfNotOverridden(systemProperties, "testGroup", testGroupId);
        putIfNotOverridden(systemProperties, "testGroup.instanceIds", String.join(",", instanceIds));
        systemProperties.putAll(getTestGroupSystemProperties(testGroupId));

        for (final Map.Entry<Integer, ProductExecution> entry : productExecutionsByWebPort.entrySet()) {
            final Product product = entry.getValue().getProduct();
            final ProductHandler productHandler = entry.getValue().getProductHandler();
            final int webPort = entry.getKey();

            if (productExecutionsByWebPort.size() == 1) {
                putIfNotOverridden(systemProperties, "http.port", String.valueOf(webPort));
                putIfNotOverridden(systemProperties, "context.path", product.getContextPath());
            }

            final String baseUrl = MavenGoals.getBaseUrl(product, webPort);
            // hard coded system properties...
            putIfNotOverridden(systemProperties,
                    "http." + product.getInstanceId() + ".port", String.valueOf(webPort));
            putIfNotOverridden(systemProperties,
                    "context." + product.getInstanceId() + ".path", product.getContextPath());
            putIfNotOverridden(systemProperties,
                    "http." + product.getInstanceId() + ".url", MavenGoals.getBaseUrl(product, webPort));
            putIfNotOverridden(systemProperties,
                    "http." + product.getInstanceId() + ".protocol", product.getProtocol());
            putIfNotOverridden(systemProperties, "baseurl." + product.getInstanceId(), baseUrl);

            // yes, this means you only get one base url if multiple products, but that is what selenium would expect
            putIfNotOverridden(systemProperties, "baseurl", baseUrl);

            putIfNotOverridden(systemProperties, "homedir." + product.getInstanceId(),
                    productHandler.getHomeDirectory(product).getAbsolutePath());
            putIfNotOverridden(systemProperties,
                    "homedir", productHandler.getHomeDirectory(product).getAbsolutePath());
            putIfNotOverridden(systemProperties, "product." + product.getInstanceId() + ".id", product.getId());
            putIfNotOverridden(systemProperties,
                    "product." + product.getInstanceId() + ".version", product.getVersion());

            systemProperties.putAll(getProductFunctionalTestProperties(product));
        }
    }

    private int setUpDebugging(final int counter, final Product product) {
        int newCounter = counter;
        if (product.getJvmDebugPort() == 0) {
            product.setJvmDebugPort(jvmDebugPort + newCounter++);
        }
        final int debugPort = product.getJvmDebugPort();
        final String debugArgs = " -Xdebug -Xrunjdwp:transport=dt_socket,address=" + debugPort +
                ",suspend=" + (jvmDebugSuspend ? "y" : "n") + ",server=y ";

        if (isBlank(product.getJvmArgs())) {
            product.setJvmArgs(trimToEmpty(jvmArgs));
        }

        product.setDebugArgs(debugArgs);
        return newCounter;
    }

    /**
     * Adds the property to the map if such property is not overridden in system properties passed to
     * maven executing this mojo.
     *
     * @param map   the properties map
     * @param key   the key to be added
     * @param value the value to be set. Will be overridden by an existing system property, if such exits.
     */
    private void putIfNotOverridden(Map<String, Object> map, String key, Object value) {
        if (!map.containsKey(key)) {
            if (System.getProperties().containsKey(key)) {
                map.put(key, System.getProperty(key));
            } else {
                map.put(key, value);
            }
        }
    }

    /**
     * Returns the classifier of the test group. Unless specified, this is "tomcat85x", the default container.
     */
    private String getClassifier(String testGroupId) {
        for (TestGroup group : getTestGroups()) {
            if (StringUtils.equals(group.getId(), testGroupId)) {
                if (group.getClassifier() != null) {
                    return group.getClassifier();
                } else {
                    return AmpsDefaults.DEFAULT_CONTAINER;
                }
            }
        }
        return AmpsDefaults.DEFAULT_CONTAINER;
    }

    private Map<String, String> getTestGroupSystemProperties(String testGroupId) {
        if (NO_TEST_GROUP.equals(testGroupId)) {
            return emptyMap();
        }

        for (TestGroup group : getTestGroups()) {
            if (StringUtils.equals(group.getId(), testGroupId)) {
                return group.getSystemProperties();
            }
        }
        return emptyMap();
    }

    private List<String> getIncludesForTestGroup(final String testGroupId) {
        if (NO_TEST_GROUP.equals(testGroupId)) {
            return singletonList(functionalTestPattern);
        } else {
            for (TestGroup group : getTestGroups()) {
                if (StringUtils.equals(group.getId(), testGroupId)) {
                    final List<String> groupIncludes = group.getIncludes();
                    if (groupIncludes.isEmpty()) {
                        return singletonList(functionalTestPattern);
                    } else {
                        return groupIncludes;
                    }
                }
            }
        }
        return singletonList(functionalTestPattern);
    }

    private List<String> getExcludesForTestGroup(String testGroupId) {
        if (NO_TEST_GROUP.equals(testGroupId)) {
            return emptyList();
        } else {
            for (TestGroup group : getTestGroups()) {
                if (StringUtils.equals(group.getId(), testGroupId)) {
                    return group.getExcludes();
                }
            }
        }
        return emptyList();
    }
}
